//
//  NSError+BRTMError.h
//  BRTMiveCore
//
//  Created by MingLQ on 2016-11-30.
//  Copyright © 2016 BaijiaYun. All rights reserved.
//

#import <Foundation/Foundation.h>

#import <BJLiveBase/BJLiveBase.h>

NS_ASSUME_NONNULL_BEGIN

/**
 BRTMError 可用属性
 .domain                    BRTMErrorDomain
 .code                      BRTMErrorCode - 错误码
 .localizedDescription      NSString * - 错误描述
 .localizedFailureReason    NSString * - 错误原因，可能为空 - TODO: 干掉，如果有具体错误信息则替换调默认的错误描述
 .bjl_sourceError           NSError * - 引起当前错误的错误，可能为空
 TODO: server errorCode, message
 */

FOUNDATION_EXPORT const NSErrorDomain BRTMErrorDomain;

typedef NS_ENUM(NSInteger, BRTMErrorCode) {
    /** 成功 */
    BRTMErrorCode_success = 0,
    //// common ////
    /** 网络错误 */
    BRTMErrorCode_networkError,
    /** 请求失败 */
    BRTMErrorCode_requestFailed,
    /** 主动调用取消 */
    BRTMErrorCode_cancelled,
    /** 非法用户角色 */
    BRTMErrorCode_invalidUserRole,
    /** 非法调用 */
    BRTMErrorCode_invalidCalling,
    /** 参数错误 */
    BRTMErrorCode_invalidArguments,
    /** 操作过于频繁 */
    BRTMErrorCode_areYouRobot,
    //// enter room ////
    /** 教室已满 */
    BRTMErrorCode_enterRoom_roomIsFull,
    /** 不支持 iOS 端进入教室 */
    BRTMErrorCode_enterRoom_unsupportedClient,
    /** 不支持当前设备进入教室 */
    BRTMErrorCode_enterRoom_unsupportedDevice,
    /** 用户被禁止进入教室 */
    BRTMErrorCode_enterRoom_forbidden,
    /** 用户已经在其它设备登录 */
    BRTMErrorCode_enterRoom_loginConflict,
    /** 试听结束 */
    BRTMErrorCode_enterRoom_auditionTimeout,
    /** 教室时间到期 */
    BRTMErrorCode_enterRoom_timeExpire,
    //// exit room ////
    /** 连接断开 */
    BRTMErrorCode_exitRoom_disconnected,
    /** 用户在其它设备登录 */
    BRTMErrorCode_exitRoom_loginConflict,
    /** 用户被请出教室 */
    BRTMErrorCode_exitRoom_kickout,
    /** 试听结束 */
    BRTMErrorCode_exitRoom_auditionTimeout,
    /** 教室时间到期 */
    BRTMErrorCode_exitRoom_timeExpire,
    /** 主讲离开，教室关闭 */
    BRTMErrorCode_exitRoom_presenterLeave,
    /* !!!: 
     1、在此之前增加错误码；
     2、不要设置错误码取值；
     3、同步增删 BRTMErrorDescriptions； */
    BRTMErrorCode_unknown    // 未知错误
};

FOUNDATION_EXPORT NSString * const BRTMErrorDescription_unknown;
FOUNDATION_EXPORT NSString * _Nonnull const BRTMErrorDescriptions[];

typedef BJLError BRTMError;

static inline BRTMError * _Nullable BRTMErrorMakeFromError(BRTMErrorCode errorCode, NSString * _Nullable reason, NSError * _Nullable sourceError) {
    if (errorCode == BRTMErrorCode_success) {
        return nil;
    }
    BRTMErrorCode titleIndex = (BRTMErrorCode)MIN(MAX(0, errorCode), BRTMErrorCode_unknown);
    NSMutableDictionary *userInfo = [NSMutableDictionary new];
    [userInfo setObject:BRTMErrorDescriptions[titleIndex] ?: BRTMErrorDescription_unknown forKey:NSLocalizedDescriptionKey];
    if (reason) {
        [userInfo setObject:reason forKey:NSLocalizedFailureReasonErrorKey];
    }
    if (sourceError) {
        [userInfo setObject:sourceError forKey:BJLErrorSourceErrorKey];
    }
    return (BRTMError *)[NSError errorWithDomain:BRTMErrorDomain code:errorCode userInfo:userInfo];
}

static inline BRTMError * _Nullable BRTMErrorMake(BRTMErrorCode errorCode, NSString * _Nullable reason) {
    return BRTMErrorMakeFromError(errorCode, reason, nil);
}

#define bjl_isRobot(LIMIT) ({ \
    static NSTimeInterval LAST = 0.0; \
    NSTimeInterval NOW = [NSDate timeIntervalSinceReferenceDate]; \
    BOOL isRobot = NOW - LAST < LIMIT; \
    if (!isRobot) { \
        LAST = NOW; \
    } \
    isRobot; \
})

#define bjl_returnIfRobot(LIMIT) { \
    static NSTimeInterval LAST = 0.0; \
    NSTimeInterval NOW = [NSDate timeIntervalSinceReferenceDate]; \
    if (NOW - LAST < LIMIT) { \
        return; \
    } \
    LAST = NOW; \
}

#define bjl_returnErrorIfRobot(LIMIT) { \
    static NSTimeInterval LAST = 0.0; \
    NSTimeInterval NOW = [NSDate timeIntervalSinceReferenceDate]; \
    if (NOW - LAST < LIMIT) { \
        return BRTMErrorMake(BRTMErrorCode_areYouRobot, \
                            [NSString stringWithFormat:@"每 %g 秒只能操作 1 次", LIMIT]); \
    } \
    LAST = NOW; \
}

#pragma mark -

@interface BRTMRobotCatcher : NSObject

@property (nonatomic) NSInteger limit;
@property (nonatomic) NSTimeInterval duration;
@property (nonatomic, readonly) NSInteger maxCatchCount, currentCatchCount, maxCount, currentCount;

+ (instancetype)robotCatcherWithLimit:(NSInteger)limit duration:(NSTimeInterval)duration;
- (nullable BRTMError *)tryCatch;

@end

NS_ASSUME_NONNULL_END
