//
//  BRTMRoom.h
//  BRTM
//
//  Created by xyp on 2020/12/21.
//

#import <Foundation/Foundation.h>
#if __has_feature(modules) && BJL_USE_SEMANTIC_IMPORT
@import BJLiveBase;
#else
#import <BJLiveBase/BJLiveBase.h>
#endif

#import "BRTMLoadingVM.h"

#import "BRTMConfig.h"
#import "BRTMRoomInfo.h"
#import "BRTMConstants.h"
#import "NSError+BRTMError.h"
#import "BRTMDocumentManager.h"
#import "BRTMChatManager.h"
#import "BRTMUserManager.h"
#import "BRTMDelegate.h"

/** 房间状态 */
typedef NS_ENUM(NSInteger, BRTMRoomState) {
    /** 初始状态、连接失败、连接断开等
     参考 `error` */
    BRTMRoomState_disconnected,
    /** 连接中 */
    BRTMRoomState_connecting,
    /** 已连接 */
    BRTMRoomState_connected
};

NS_ASSUME_NONNULL_BEGIN

@interface BRTMRoom : NSObject

/// 设置appID
/// @param appID
/// 需要先设置appID, 否则调用 createRoom 实例化会失败
+ (void)setAppID:(NSString *)appID;

/// 创建一个room, 不是单例
/// 如果没有设置appID, 返回nil
+ (nullable instancetype)createRoom;

/// 进入房间相关的代理
@property (nonatomic, weak) id <BRTMDelegate> delegate;

#pragma mark lifecycle
@property (nonatomic, readonly) BRTMRoomState state;
@property (nonatomic, readonly, nullable) BRTMError *error;
@property (nonatomic, readonly) BRTMConfig *config;

/// 进入房间
/// @param config 进入房间配置
/// BRTMErrorCode_invalidArguments  参数出错
- (nullable BRTMError *)joinRoomWithConfig:(BRTMConfig *)config;

/// 教室信息, 内部【不】读取此处 `roomInfo`
@property (nonatomic, readonly, copy, nullable) NSObject<BRTMRoomInfo> *roomInfo;

/// 断开、重连
/// @param reloadingBlock 重连回调。reloadingVM：重连 vm；callback(reload)：调用 callback 时 reload 参数传 YES 重连，NO 将导致 `异常退出`
/// 网络连接断开时回调，回调 callback 确认是否重连，YES 重连、NO 退出教室，也可延时或者手动调用 callback
/// 可通过 `reloadingVM` 监听重连的进度和结果
/// 默认（不设置此回调）在断开时自动重连、重连过程中遇到错误将 `异常退出`
/// !!!: 断开重连过程中 vm 的状态、数据没有与服务端同步，调用其它 vm 方法时发起的网络请求会被丢弃、甚至产生不可预期的错误
- (void)setReloadingBlock:(void (^ _Nullable)(BRTMLoadingVM *reloadingVM,
                                              void (^callback)(BOOL reload)))reloadingBlock;

/// 退出房间
- (void)leaveRoom;

#pragma mark metainfo

/// 当前登录用户信息
/// !!!: 内部【不】读取此处 `loginUser`
@property (nonatomic, readonly, copy, nullable) BJLUser *loginUser;

#pragma mark view & view-model

/// 进教室的 loading 状态，参考 `BRTMLoadingVM`
@property (nonatomic, readonly, nullable) BRTMLoadingVM *loadingVM;

/// 禁用动画课件
/// 是否禁用动画课件由两个参数控制，任意一个值为 YES 就禁止
/// 1、由服务端通过 `BRTMFeatureConfig` 的 `disablePPTAnimation` 控制
/// 2、由上层设置这个 `disablePPTAnimation`
@property (nonatomic) BOOL disablePPTAnimation;

/// 用户相关, 详细参阅 BRTMUserManager.h
@property (nonatomic, readonly, nullable) BRTMUserManager *userManager;

/// 聊天相关, 详细参阅 BRTMChatManager.h
@property (nonatomic, readonly, nullable) BRTMChatManager *chatManager;

/// 消息相关的代理. note: 区别于 chat
@property (nonatomic, weak) id <BRTMMessageDelegate> messageDelegate;

#pragma mark - deployment

/// 环境部署，需要在进教室之前设置
/// !!!: 仅供内部使用，集成 SDK 的用户请勿使用
@property (class, nonatomic) BRTMDeployType deployType;

@end

@interface BRTMRoom (signal)

/// 发送自定义广播信令, 通过'messageDelegate' 的 'room:didReceiveBoardcastMessageWithKey:value:' 监听
/// @param key 自定义的key
/// @param value 自定义的发送内容
/// @param cache 是否需要缓存
/// BRTMErrorCode_invalidArguments  参数不对
/// BRTMErrorCode_areYouRobot       发送频率过快，要求每秒不超过 5 条、并且每分钟不超过 60 条
- (nullable BRTMError *)sendBroadcastMessageWithKey:(NSString *)key value:(id)value cache:(BOOL)cache;

/// 请求自定义广播的缓存, 通过'messageDelegate' 的 'room:didReceiveBoardcastMessageWithKey:value:' 监听
/// @param key 自定义的key
- (nullable BRTMError *)requestBroadcastMessageCacheWithKey:(NSString *)key;

/// 清除定制广播信令缓存
/// @param key 自定义的key
- (nullable BJLError *)requestCustomizedBroadcastCacheClearWithKey:(NSString *)key;

/// 请求自定义广播的缓存, 通过'messageDelegate' 的 'room:didReceiveMessageWithKey:value:fromUserID:' 监听
/// @param key key description
/// @param value value description
/// @param userID 指定用户的userID
/// BRTMErrorCode_invalidArguments  参数不对
/// BRTMErrorCode_areYouRobot       发送频率过快，要求每秒不超过 5 条、并且每分钟不超过 60 条
- (nullable BRTMError *)sendMessageWithKey:(NSString *)key value:(id _Nullable)value toUserWithUserID:(NSString *)userID;

@end

#pragma mark - 开始/结束文档操作

@interface BRTMRoom (document)

/// 课件管理、显示、控制，画笔管理等, 参考 `documentManager`
/// #discussion 默认为nil, 使用白板的时候需要调用 createDocumentManager 进行实例化
@property (nonatomic, readonly, nullable) BRTMDocumentManager *documentManager;

/// 启用文档服务
- (void)enabelDocumentService;

/// 停止文档服务, 会销毁 documentManager 实例
- (void)disabelDocumentService;

/// 实例化 BRTMDocumentManager,  调用此方法后 documentAdatper 才有值
- (void)createDocumentManager;

/// 销毁 BRTMDocumentManager  实例
- (void)destoryDocumentManager;

@end

NS_ASSUME_NONNULL_END
