//
//  BRTMDocumentVM.h
//  BRTMLiveCore
//
//  Created by HuangJie on 2018/9/13.
//  Copyright © 2018年 BaijiaYun. All rights reserved.
//

#if __has_feature(modules) && BJL_USE_SEMANTIC_IMPORT
@import BJLiveBase;
#else
#import <BJLiveBase/BJLiveBase.h>
#endif

#import "BRTMBaseVM.h"

NS_ASSUME_NONNULL_BEGIN

extern NSString * const BRTMBlackboardID;

/**
 ### 课件管理、显示
 #discussion BRTMDocument：课件，教室内默认至少有一个白板课件，老师可能会上传多个课件，每个课件有一页或多页，每个课件内的 pageIndex 单独计算、从 0 开始
 #discussion BJLSlidePage：幻灯片，将所有课件拆散、组成成幻灯片序列，课件的每一页对应一个幻灯片，所有幻灯片 slidePageIndex 整体计算、从 0 开始
 #discussion 参考 `loadAllDocuments`
 */
@interface BRTMDocumentVM : BRTMBaseVM

#pragma mark - document management

/**
 本地翻页, 远端不同步,  default: NO
 disabeChangePage 为 YES 时, 调用 requestTurnToDocumentID:pageIndex 会返回 BRTMErrorCode_invalidUserRole
 */
@property (nonatomic) BOOL disabeChangePage;

/**
 翻页课件
 #param documentID 课件 ID
 #param pageIndex 目标页在课件中的序号
 #return BRTMError:
 BRTMErrorCode_invalidArguments  错误参数
 BRTMErrorCode_invalidUserRole 当前用那个户不能翻页
 */
- (nullable BJLError *)requestTurnToDocumentID:(NSString *)documentID
                                     pageIndex:(NSInteger)pageIndex;

/**
 添加课件
 #discussion 添加成功将调用 `BJLDocumentVM` 的 `didAddDocument:`
 #param document 课件
 #return BJLError:
 BRTMErrorCode_invalidArguments  错误参数
 */
- (nullable BJLError *)addDocument:(BJLDocument *)document;

/**
 删除课件
 #discussion 删除成功将调用 `BJLDocumentVM` 的 `didDeleteDocument:`
 #param documentID 课件 ID
 #return BJLError:
 BRTMErrorCode_invalidArguments  错误参数
 */
- (nullable BJLError *)deleteDocumentWithID:(NSString *)documentID;

/**
 添加白板
 #discussion 最多同时存在 10 页白板
 #return BJLError
 */
- (nullable BJLError *)addWhiteboardPage;

/**
 删除白板
 #param pageIndex 白板页码，使用 BJLSlidePage 的 `slidePageIndex`
 #return BJLError
 */
- (nullable BJLError *)deleteWhiteboardPageWithIndex:(NSInteger)pageIndex;

/**
 上传图片，用于添加课件
 #param fileURL     图片文件路径
 #param progress    上传进度，非主线程回调、可能过于频繁
 - progress         0.0 ~ 1.0
 #param finish      结束回调：
 - document         非 nil 即为成功，用于 `addDocument:`
 - error            错误
 #return            upload task
 */
- (NSURLSessionUploadTask *)uploadImageFile:(NSURL *)fileURL
                                   progress:(nullable void (^)(CGFloat progress))progress
                                     finish:(void (^)(BJLDocument * _Nullable document, BJLError * _Nullable error))finish;

/**
 上传文档
 #discussion 图片作为文档上传没有转码过程，其余的PPT等文档需要等待转码过程
 #discussion 转码服务器主动进行，不需要调用其他接口
 #param fileURL 文档路径
 #param mimeType 文档类型
 #param fileName 文件名
 #param isAnimated 是否有动画
 #param progress 进度 0.0 ~ 1.0
 #param finish 结束回调
 #return task
 */
- (NSURLSessionUploadTask *)uploadFile:(NSURL *)fileURL
                              mimeType:(NSString *)mimeType
                              fileName:(NSString *)fileName
                            isAnimated:(BOOL)isAnimated
                              progress:(nullable void (^)(CGFloat progress))progress
                                finish:(void (^)(BJLDocument * _Nullable document, BJLError * _Nullable error))finish;

/**
 文档转码进度
 #discussion 可以通过这个接口查询文档转码进度
 #discussion 转码完成后，需要去获取转码后的新的文档
 #param fileIDList file ID 数组
 #param completion transcodeArray
 #return task
 */
- (nullable NSURLSessionDataTask *)requestTranscodingProgressWithFileIDList:(NSArray<NSString *> *)fileIDList
                                                                 completion:(nullable void (^)(NSArray<BJLDocumentTranscodeModel *> * _Nullable transcodeModelArray, BJLError * _Nullable error))completion;

/**
 获取文档转码后的新的文档
 #discussion 根据文档ID获取转码后的新的文档数据
 #discussion 需要将获取到的文档列表添加到教室
 #param fileIDList file ID 数组
 #param completion BJLDocument 数组
 #return task
 */
- (nullable NSURLSessionDataTask *)requestDocumentListWithFileIDList:(NSArray<NSString *> *)fileIDList
                                                          completion:(nullable void (^)(NSArray<BJLDocument *> * _Nullable documentArray, BJLError * _Nullable error))completion;

/**
 获取文档备注信息
 #param documentID 文档ID
 #param completion 返回 remarkInfo 备注信息
 #return task
 */
- (nullable NSURLSessionDataTask *)requestCompleteRemarkInfoWithDocumentID:(NSString *)documentID
                                                                completion:(void (^)(NSDictionary * _Nullable remarkInfo, BJLError * _Nullable error))completion;

#pragma mark - document show

/** 所有课件 */
@property (nonatomic, readonly, copy, nullable) NSArray<BJLDocument *> *allDocuments;


/** 设置单实例文档下的白板数据，进教室前设置，未设置使用默认白板数据 */
@property (nonatomic, nullable) BJLWhiteboard *whiteboard;

/**
 `allDocuments` 被覆盖更新
 #discussion 覆盖更新才调用，增量更新不调用
 #param allDocuments 所有课件
 */
- (BJLObservable)allDocumentsDidOverwrite:(nullable NSArray<BJLDocument *> *)allDocuments currentDocumentID:(NSString *)documentID;

/**
 加载所有课件
 #discussion 连接教室后、掉线重新连接后自动调用加载
 #discussion 加载成功后更新 `allDocuments`、调用 `allDocumentsDidOverwrite:`
 */
- (void)loadAllDocuments;

/**
 添加课件通知
 #discussion 同时更新 `allDocuments`
 #param document 课件
 */
- (BJLObservable)didAddDocument:(BJLDocument *)document;

/**
 删除课件通知
 #discussion 同时更新 `allDocuments`
 #param document 课件
 */
- (BJLObservable)didDeleteDocument:(BJLDocument *)document;

/**
 添加白板通知
 #discussion 同时更新 `allDocuments`
 #param pageIndex 白板页码
 */
- (BJLObservable)didAddWhiteboardPage:(NSInteger)pageIndex document:(BJLDocument *)doctment;

/**
 删除白板通知
 #discussion 同时更新 `allDocuments`
 #param pageIndex 白板页码
 */
- (BJLObservable)didDeleteWhiteboardPageWithIndex:(NSInteger)pageIndex document:(BJLDocument *)doctment;

/** 课件总页数，未知页数的文档认为只有一页 */
@property (nonatomic, readonly) NSInteger totalPageCount;

/** 课件当前页信息 */
@property (nonatomic, readonly, nullable) BJLSlidePage *currentSlidePage;

/**
 通过 documentID 获取 document
 #param documentID 课件 ID
 */
- (nullable BJLDocument *)documentWithID:(NSString *)documentID;

/**
 通过 documentID、pageIndex 获取 slide page
 #param documentID 课件 ID
 #param pageIndex 目标页在课件中的序号
 */
- (nullable BJLSlidePage *)slidePageWithDocumentID:(NSString *)documentID
                                         pageIndex:(NSInteger)pageIndex;

/**
 通过 documentID、pageIndex 获取 slide page
 #param documentID 课件 ID
 #param pageIndex 目标页在课件中的序号
 #param step 具体动画步数
 */
- (nullable BJLSlidePage *)slidePageWithDocumentID:(NSString *)documentID
                                         pageIndex:(NSInteger)pageIndex
                                              step:(NSInteger)step;

/**
 指定 documentID，获取对应课件的所有页面
 #param documentID 课件 ID
 */
- (NSArray<BJLSlidePage *> *)slidePagesWithDocumentID:(NSString *)documentID;

/** 获取 slide pages */
- (NSArray<BJLSlidePage *> *)allSlidePages;

#pragma mark - interactive class 专业版小班课 API

/** 黑板视图控制器 */
@property (nonatomic, readonly) UIViewController<BJLBlackboardUI> *blackboardViewController;
/** 黑板页数 */
@property (nonatomic, readonly) NSInteger blackboardContentPages;

/** 设置黑板视图（blackboardViewController）的默认背景图 */
@property (nonatomic) UIImage *blackboardImage;

/**
 指定文档 ID 创建对应的视图控制器
 #param documentID 文档ID，通过 BJLDocument 的 documentID 获得
 #return 对应文档的视图控制器
 */
- (UIViewController<BJLSlideshowUI> *)documentViewControllerWithID:(NSString *)documentID;

/**
 更新文档显示信息
 #param documentID 文档 ID
 #param displayInfo 文档显示信息
 */
- (nullable BJLError *)updateDocumentWithID:(NSString *)documentID displayInfo:(BJLDocumentDisplayInfo *)displayInfo;

/**
 更新文档显示信息的通知
 #param document 更新后的文档信息
 */
- (BJLObservable)didUpdateDocument:(BJLDocument *)document;

/**
 更新文档显示信息的通知，同`didUpdateDocument`，但是语意更明显
 #param documentDisplayInfo 文档的显示信息
 #param document 更新后的文档信息
 */
- (BJLObservable)displayInfoDidUpdate:(BJLDocumentDisplayInfo *)documentDisplayInfo document:(BJLDocument *)document;

/**
 设置白板的背景图
 #param imageUrl 图片链接
 */
- (void)setWhiteboardBackgroundImageUrl:(NSString *)imageUrl;

@end

NS_ASSUME_NONNULL_END
