//
//  BRTMDocumentManager.h
//  BRTM
//
//  Created by xyp on 2020/12/12.
//  Copyright © 2020 BaijiaYun. All rights reserved.
//

#import <Foundation/Foundation.h>
#if __has_feature(modules) && BJL_USE_SEMANTIC_IMPORT
@import BJLiveBase;
#else
#import <BJLiveBase/BJLiveBase.h>
#endif

#import "NSError+BRTMError.h"
#import "BRTMConstants.h"

#import "BRTMDocumentDelegate.h"

NS_ASSUME_NONNULL_BEGIN

extern NSString * const kBRTMBlackboardID;

/**
 ### 课件管理、显示
 #discussion BRTMDocument：课件，教室内默认至少有一个白板课件，老师可能会上传多个课件，每个课件有一页或多页，每个课件内的 pageIndex 单独计算、从 0 开始
 #discussion BJLSlidePage：幻灯片，将所有课件拆散、组成成幻灯片序列，课件的每一页对应一个幻灯片，所有幻灯片 slidePageIndex 整体计算、从 0 开始
 */

@interface BRTMDocumentManager : NSObject

- (instancetype)init NS_UNAVAILABLE;

/// 文档和画笔相关的代理
@property (nonatomic, weak) id <BRTMDocumentDelegate> documentDelegate;

/// 所有课件, allDocuments发生变化会回调 'BRTMDocumentDelegate' 的 'allDocumentsDidOverwrite'
@property (nonatomic, readonly, copy, nullable) NSArray<BJLDocument *> *allDocuments;

///  本地翻页, 远端不同步,  default: NO
/// disabeChangePage 为 YES 时, 调用 requestTurnToDocumentID:pageIndex 会返回 BRTMErrorCode_invalidUserRole
@property (nonatomic) BOOL disabeChangePage;

/// 翻页课件
/// @param documentID 课件 ID
/// @param pageIndex 目标页在课件中的序号, 即页码
/// 成功会回调 'BRTMDocumentDelegate' 的 'didPageChange'
- (nullable BRTMError *)requestTurnToDocumentID:(NSString *)documentID pageIndex:(NSInteger)pageIndex;

/// 增加文档
/// @param document document
/// 成功将回调 'BRTMDocumentDelegate' 的 'didAddDocument'
- (nullable BRTMError *)addDocument:(BJLDocument *)document;

/// 删除文档
/// @param documentID documentID description
/// 成功将回调 'BRTMDocumentDelegate' 的 'didAddDocument'
- (nullable BRTMError *)deleteDocumentWithID:(NSString *)documentID;

/// 增加一页白板,  成功将回调 'BRTMDocumentDelegate' 的 'didAddWhiteboardPage'
/// 只有白板的document才有效
/// 限制每 0.5s 只能调用一次
- (nullable BRTMError *)addPageToDocument:(BJLDocument *)document;

/// 删除一页白板,  只有白板的document才有效
/// @param pageIndex 白板页码
/// 成功将回调 'BRTMDocumentDelegate' 的 'didDeleteWhiteboardPageWithIndex'
/// 限制每 0.5s 只能调用一次
- (nullable BRTMError *)deletePageIndex:(NSInteger)pageIndex document:(BJLDocument *)document;

/// 课件当前页信息
@property (nonatomic, readonly, nullable) BJLSlidePage *currentSlidePage;

/// 通过documentID获取document
/// @param documentID documentID description
- (nullable BJLDocument *)documentWithID:(NSString *)documentID;

/// 给白板设置背景图片
/// @param imageUrl imageUrl
- (void)setWhiteboardBackgroundImageUrl:(NSString *)imageUrl;

/// 指定文档 ID 创建对应的视图控制器
/// @param documentID 指定的documentID
- (UIViewController<BJLSlideshowUI> *)documentViewControllerWithID:(NSString *)documentID;

/// 上传图片，用于添加课件
/// @param fileURL 图片文件路径
/// @param progress 上传进度，非主线程回调、可能过于频繁, 0.0 ~ 1.0
/// @param finish 结束回调：document: 非 nil 即为成功，用于 `addDocument:` ; error: 错误
- (NSURLSessionUploadTask *)uploadImageFile:(NSURL *)fileURL
                                   progress:(nullable void (^)(CGFloat progress))progress
                                     finish:(void (^)(BJLDocument * _Nullable document, BRTMError * _Nullable error))finish;

/// 上传文档
/// 图片作为文档上传没有转码过程，其余的PPT等文档需要等待转码过程, 转码服务器主动进行，不需要调用其他接口
/// @param fileURL  文档路径
/// @param mimeType 文档类型
/// @param fileName 文件名
/// @param isAnimated 是否有动画
/// @param progress  进度 0.0 ~ 1.0
/// @param finish 结束回调
- (NSURLSessionUploadTask *)uploadFile:(NSURL *)fileURL
                              mimeType:(NSString *)mimeType
                              fileName:(NSString *)fileName
                            isAnimated:(BOOL)isAnimated
                              progress:(nullable void (^)(CGFloat progress))progress
                                finish:(void (^)(BJLDocument * _Nullable document, BRTMError * _Nullable error))finish;

/// 文档转码进度查询
/// 可以通过这个接口查询文档转码进度, 转码完成后，需要去获取转码后的新的文档
/// @param fileIDList file ID 数组
/// @param completion transcodeArray
- (nullable NSURLSessionDataTask *)requestTranscodingProgressWithFileIDList:(NSArray<NSString *> *)fileIDList
                                                                 completion:(nullable void (^)(NSArray<BJLDocumentTranscodeModel *> * _Nullable transcodeModelArray, BRTMError * _Nullable error))completion;

/// 获取文档转码后的新的文档
/// 根据文档ID获取转码后的新的文档数据, 需要将获取到的文档列表添加到房间
/// @param fileIDList file ID 数组
/// @param completion Document 数组
- (nullable NSURLSessionDataTask *)requestDocumentListWithFileIDList:(NSArray<NSString *> *)fileIDList
                                                          completion:(nullable void (^)(NSArray<BJLDocument *> * _Nullable documentArray, BRTMError * _Nullable error))completion;

/// 获取文档备注信息
/// @param documentID 文档ID
/// @param completion 返回 remarkInfo 备注信息
- (nullable NSURLSessionDataTask *)requestCompleteRemarkInfoWithDocumentID:(NSString *)documentID
                                                                completion:(void (^)(NSDictionary * _Nullable remarkInfo, BRTMError * _Nullable error))completion;


#pragma mark - 画笔相关的

/// 画笔操作模式
@property (nonatomic, readonly) BJLBrushOperateMode brushOperateMode;

/// 画笔类型
@property (nonatomic) BJLDrawingShapeType shapeType;

/// 画笔边框颜色
@property (nonatomic, nonnull) NSString *strokeColor;

/// 画笔边框颜色透明度，取值范围 0~1
@property (nonatomic) CGFloat strokeAlpha;

/// 画笔填充颜色
@property (nonatomic, nullable) NSString *fillColor;

/// 画笔填充颜色透明度，取值范围 0~1, fillColor 不为空时有效
@property (nonatomic) CGFloat fillAlpha;

/// doodle 画笔线宽
@property (nonatomic) CGFloat doodleStrokeWidth;

/// 图形画笔边框线宽
@property (nonatomic) CGFloat shapeStrokeWidth;

/// 文字画笔字体大小,  范围 [0.0001, 1.0], 默认值0.02
/// 渲染文字时的 字体 为 textFontSize * 画布宽度
@property (nonatomic) CGFloat textFontSize;

/// 文字画笔是否加粗
@property (nonatomic) BOOL textBold;

/// 文字画笔是否为斜体
@property (nonatomic) BOOL textItalic;

/// 画笔开关状态，参考 `updatePaintEnabled:`
@property (nonatomic, readonly) BOOL paintEnabled;

/// doodle 画笔是否虚线
@property (nonatomic) BOOL isDottedLine;

/// 选中画笔时是否显示归属信息
@property (nonatomic) BOOL showShapeAuthorNameWhenSelected;

/// 是否有选中的画笔
@property (nonatomic) BOOL hasSelectedShape;

/// 开启、关闭画笔
/// @param paintEnabled YES：开启，NO：关闭
/// 开启画笔时，单文档实例情况下如果本地页数与服务端页数不同步则无法绘制
- (nullable BRTMError *)updatePaintEnabled:(BOOL)paintEnabled;

/// 更新画笔操作模式
/// @param operateMode 操作模式
/// BRTMErrorCode_invalidCalling paintEnabled 是 NO
- (nullable BRTMError *)updateBrushOperateMode:(BJLBrushOperateMode)operateMode;

///// 添加图片画笔
///// @param imageURL 图片 url
///// @param relativeFrame 图片相对于画布的 frame, 各项数值取值范围为 [0.0, 1.0]
///// @param documentID 目标文档 ID
///// @param pageIndex 目标页
///// @param isWritingBoard 是否为小黑板
//- (nullable BRTMError *)addImageShapeWithURL:(NSString *)imageURL
//                              relativeFrame:(CGRect)relativeFrame
//                               toDocumentID:(NSString *)documentID
//                                  pageIndex:(NSUInteger)pageIndex
//                             isWritingBoard:(BOOL)isWritingBoard;

/// 清除某一页的所有画笔
/// @param documentID 文档 ID
/// @param pageIndex 当前页的id
/// @param userID userID
- (nullable BRTMError *)clearShapesWithDocumentID:(NSString *)documentID
                                        pageIndex:(NSUInteger)pageIndex
                                           userID:(nullable NSString *)userID;


#pragma mark - 激光笔

/// 改变内部实现的激光笔的图片
- (void)updateLaserPointImage:(UIImage *)image;

/// 自定义激光笔
/// 默认使用内部实现的激光笔，如果调用这个方法，则需要外部实现接收激光笔点的ui
- (void)customLaserPonitUI;

/// 激光笔位置移动请求
/// @param location 激光笔目标位置
/// NOTE: 自定义激光笔时调用
- (nullable BRTMError *)moveLaserPointToLocation:(CGPoint)location
                                      documentID:(nonnull NSString *)documentID
                                       pageIndex:(NSUInteger)pageIndex;

@end

NS_ASSUME_NONNULL_END
