//
//  BRTMChatVM.h
//  BRTMLiveCore
//
//  Created by MingLQ on 2016-12-10.
//  Copyright © 2016 BaijiaYun. All rights reserved.
//

#if __has_feature(modules) && BJL_USE_SEMANTIC_IMPORT
@import BJLiveBase;
#else
#import <BJLiveBase/BJLiveBase.h>
#endif

#import "BRTMBaseVM.h"

NS_ASSUME_NONNULL_BEGIN

/** ### 聊天 */
@interface BRTMChatVM : BRTMBaseVM

/** fromUser 能否向用户 toUser 发送私聊消息 */
- (BOOL)canSendPrivateMessageFromeUser:(BJLUser *)fromUser toUser:(BJLUser *)toUser;

#pragma mark - receive message

/**
 收到历史消息 - 需要覆盖更新本地缓存的消息
 #discussion 首次进教室或断开重连会收到此回调
 #param receivedMessages 收到的所有消息
 */
- (BJLObservable)receivedMessagesDidOverwrite:(nullable NSArray<BJLMessage *> *)receivedMessages;

/**
 收到消息
 #param messages 消息
 */
- (BJLObservable)didReceiveMessages:(NSArray<BJLMessage *> *)messages;

#pragma mark - send message

/**
 发送文字消息，参考 `sendMessage:channel:` 方法
 #param text 消息，不能是空字符串或 nil
 #return BJLError:
 BRTMErrorCode_invalidArguments  参数错误；
 BRTMErrorCode_invalidCalling    错误调用，如禁言状态时调用此方法发送消息。
 */
- (nullable BJLError *)sendMessage:(NSString *)text;

/**
 指定对象，发送文字消息
 #param text 文字消息内容
 #param user 发送对象
 #return BJLError:
 BRTMErrorCode_invalidArguments  参数错误；
 BRTMErrorCode_invalidCalling    错误调用，如禁言状态时调用此方法发送消息。
 */
- (nullable BJLError *)sendMessage:(NSString *)text toUser:(nullable BJLUser *)user;

/**
 指定频道，发送文字消息
 #discussion 最多 BJLTextMaxLength_chat 个字符
 #discussion 成功后会收到消息通知
 #discussion 学生在禁言状态不能发送消息，参考 `forbidMe`、`forbidAll`
 #discussion 参考 `BJLMessage`
 #param text 消息，不能是空字符串或 nil
 #param channel 频道
 */
- (nullable BJLError *)sendMessage:(NSString *)text channel:(nullable NSString *)channel;

/**
 发送图片、表情等其它类型的消息
 #discussion 发送图片需事先调用 `uploadImageFile:progress:finish:` 方法上传
 #discussion 发送表情和图片需将 `(NSString *)emoticon.key` 或 `(NSString *)imageURLString` 转换成 (NSDictionary *)data，参考 `BJLMessage` 的 `messageDataWithEmoticonKey:` 和 `messageDataWithImageURLString:imageWidth:imageHeight:` 方法
 #param data 消息内容
 */
- (nullable BJLError *)sendMessageData:(NSDictionary *)data;

/**
 指定对象，发送图片、表情等其它类型的消息
 #param data 消息数据
 #param user 发送对象
 */
- (nullable BJLError *)sendMessageData:(NSDictionary *)data toUser:(nullable BJLUser *)user;

/**
 指定频道，发送图片、表情等其它类型的消息
 #param data 消息内容
 #param channel 频道
 */
- (nullable BJLError *)sendMessageData:(NSDictionary *)data channel:(nullable NSString *)channel;

/**
 上传图片，用于发送消息
 #param fileURL     图片文件路径
 #param progress    上传进度，非主线程回调、可能过于频繁
 - progress         0.0 ~ 1.0
 #param finish      结束
 - imageURLString   非 nil 即为成功
 - error            错误
 #return            upload task
 */
- (NSURLSessionUploadTask *)uploadImageFile:(NSURL *)fileURL
                                   progress:(nullable void (^)(CGFloat progress))progress
                                     finish:(void (^)(NSString * _Nullable imageURLString, BJLError * _Nullable error))finish;

#pragma mark - revoke message

/**
 撤回消息
 #discussion 老师可以撤回任意用户消息，学生可以撤回自己消息
 #param BJLMessage 聊天消息
 #return error
 */
- (nullable BJLError *)revokeMessage:(BJLMessage *)message;

/**
 撤回消息成功
 #param messageID 消息ID
 #param isCurrentUserRevoke 是否是当前用户撤回的消息
 */
- (BJLObservable)didRevokeMessageWithID:(NSString *)messageID isCurrentUserRevoke:(BOOL)isCurrentUserRevoke;

#pragma mark - translate message

/**
 聊天消息英汉互译，每秒最多仅允许发送一条
 #param message 消息
 #param messageUUID 用于区分消息的唯一ID,不建议使用BJLMessage.ID, 不保证其唯一性
 #param translateType 英->汉 or 汉->英
 */
- (nullable BJLError *)translateMessage:(BJLMessage *)message
                            messageUUID:(nullable NSString *)messageUUID
                          translateType:(BJLMessageLanguageType)translateType;

/**
 收到翻译结果
 #param translation 消息翻译结果
 #param messageUUID 消息唯一ID
 */
- (BJLObservable)didReceiveMessageTranslation:(NSString *)translation
                                  messageUUID:(nullable NSString *)messageUUID;


@end

NS_ASSUME_NONNULL_END
