//
//  BRTCCoreOCDelegate.h
//  BRTCCoreOC
//
//  Created by lw0717 on 2023/4/26.
//

#ifndef BRTCCoreOCDelegate_h
#define BRTCCoreOCDelegate_h

#import <Foundation/Foundation.h>
#import "BRTCCoreOCDef.h"

NS_ASSUME_NONNULL_BEGIN
@protocol BRTCCoreOCDelegate <NSObject>
@optional

/////////////////////////////////////////////////////////////////////////////////
//
//                      错误事件和警告事件
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * 错误事件回调：不可恢复的错误，一定要监听。
 *
 * @param errCode 错误码，详情见 BRTCCoreAVError 定义
 * @param errMsg 错误信息
 * @param extraInfo 扩展信息字段，个别错误码可能会带额外的信息帮助定位问题
 */
- (void)onError:(BRTCCoreOCAVError)errCode
         errMsg:(NSString *)errMsg
        extInfo:(NSDictionary *)extInfo;

 /**
  * @brief 警告事件回调
  *
  * @param warningCode 警告码，详情见 BRTCCoreAVWarning 定义
  * @param warningMsg 警告信息
  * @param extraInfo 扩展信息字段，个别警告码可能会带额外的信息帮助定位问题
  */
- (void)onWarning:(BRTCCoreOCAVWarning)warningCode
       warningMsg:(NSString *)warningMsg
          extInfo:(NSDictionary *)extInfo;

//
- (void)onBliveEnterRoom:(int)result
                taskList:(BRTCCoreOCBliveTaskList *)taskList
              roomConfig:(BRTCCoreOCBliveRoomConfig *)roomConfig;

- (void)onBliveExitRoom:(int)reason;

- (void)onBliveConnectionLost;

- (void)onBliveTryToReconnect;

- (void)onBliveConnectionRecovery;

- (void)onMixStreamAdded:(NSString *)streamId 
              streamProp:(BRTCCoreOCBliveMixStreamProperty *)streamProp;

- (void)onMixStreamRemoved:(NSString *)roomId 
                  streamId:(NSString *)streamId 
                customData:(NSString *)customData;

- (void)onMixStreamUpdate:(NSString *)streamId 
               streamProp:(BRTCCoreOCBliveMixStreamProperty *)streamProp;

- (void)onMixedNotification:(NSString *)streamId 
                 customData:(NSString *)customData;

- (void)onBliveConflictLogin;

- (void)onStartTranscode:(NSString *)streamId 
                 errCode:(BRTCCoreOCAVError)errCode
                  errMsg:(NSString *)errMsg;

- (void)onStopTranscode:(NSString *)streamId 
                errCode:(BRTCCoreOCAVError)errCode
                 errMsg:(NSString *)errMsg;

- (void)onBliveSwitchRole:(BRTCCoreOCBliveRole)role;
 
/**
 * @brief 进入房间成功与否的事件回调
 *
 * @param result result > 0 时为进房耗时（ms），result < 0
 * 时为进房错误码，错误码参见 BRTCCoreAVError
 */
- (void)onEnterRoom:(int)result;

/**
 * @brief 离开房间的事件回调
 *
 * @param reason 离开房间原因，0：主动调用 exitRoom
 * 退出房间；1：被服务器踢出当前房间；2：当前房间整个被解散
 */
- (void)onExitRoom:(int)reason;

/**
 * @brief 切换角色的事件回调
 *
 * @param errCode 错误码，BRTCCoreOCAVError_Null 代表切换成功，其他请参见 BRTCCoreOCAVError
 * @param errMsg 错误信息
 */
- (void)onSwitchRole:(BRTCCoreOCAVError)errCode errMsg:(NSString *)errMsg;

/**
 * @brief 查询用户信息回调
 *
 * @param inputUserIds 查询的用户 ID 数组
 * @param retUserIds 返回的用户 ID 数组
 */
- (void)onQueryUser:(NSArray *)inputUserIds outputUserIds:(NSArray *)outputUserIds;

/**
 * @brief 切换房间的结果回调
 *
 * @param errCode 错误码，BRTCCoreOCAVError_Null 代表切换成功，其他请参见 BRTCCoreOCAVError
 * @param errMsg 错误信息
 */
- (void)onSwitchRoom:(BRTCCoreOCAVError)errCode errMsg:(NSString *)errMsg;

/**
 * @brief 请求跨房通话的结果回调
 *
 * @param userId 要跨房通话的另一个房间中的用户 ID
 * @param errCode 错误码，BRTCCoreOCAVError_Null 代表切换成功，其他请参见 BRTCCoreOCAVError
 * @param errMsg 错误信息
 */
- (void)onConnectOtherRoom:(NSString *)userId
                   errCode:(BRTCCoreOCAVError)errCode
                    errMsg:(NSString *)errMsg;

/**
 * @brief 结束跨房通话的结果回调
 *
 * @param errCode 错误码，BRTCCoreOCAVError_Null 代表成功，其他请参见 BRTCCoreOCAVError
 * @param errMsg 错误信息
 */
- (void)onDisconnectOtherRoom:(BRTCCoreOCAVError)errCode
                       errMsg:(NSString *)errMsg;

/**
 * @brief 收到自定义消息的事件回调
 *
 * @param userId 用户标识
 * @param cmdID 命令 ID
 * @param seq 消息序号
 * @param message 消息数据
 */
- (void)onRecvCustomCmdMsg:(NSString *)userId
                     cmdID:(int)cmdID
                       seq:(int)seq
                   message:(NSData *)message;

/**
 * @brief 自定义消息丢失的事件回调
 * 只有在发送端设置了可靠传输（reliable），接收方才能收到消息的丢失回调
 *
 * @param userId 用户标识
 * @param cmdID 命令 ID
 * @param errCode 错误码
 * @param missed 丢失的消息数量
 */
- (void)onMissCustomCmdMsg:(NSString *)userId
                     cmdID:(int)cmdID
                   errCode:(int)errCode
                    missed:(int)missed;

/**
 * @brief 收到 SEI 消息的回调
 *
 * @param userId 用户标识
 * @param message 数据
 */
- (void)onRecvSEIMsg:(NSString *)userId message:(NSData *)message;

/**
 * @brief 有远端用户加入当前房间
 * 在大房间类型下，只有用户推流了才会触发远端用户进房这个回调
 * 进房前已经在房间里的远端用户和进房后才进房的远端用户都会在这个回调事件通知
 *
 * @param userId 远端用户的用户标识
 */
- (void)onRemoteUserEnterRoom:(NSString *)userId;

/**
 * @brief 有远端用户离开当前房间
 *
 * @param userId 远端用户的用户标识
 * @param reason 离开原因，0 表示用户主动退出房间，1 表示被踢出房间
 */
- (void)onRemoteUserLeaveRoom:(NSString *)userId reason:(int)reason;

/**
 * @brief 某远端用户发布/取消了主路视频画面
 *
 * @param userId 远端用户的用户标识
 * @param available
 * 该用户是否发布（或取消发布）了主路视频画面，true：发布；false：取消发布
 */
- (void)onUserVideoAvailable:(NSString *)userId available:(bool)available;

/**
 * @brief 某远端用户发布/取消了辅路视频画面
 *
 * @param userId 远端用户的用户标识
 * @param available 该用户是否发布（或取消发布）了辅路视频画面，true:
 * 发布；false：取消发布
 */
- (void)onUserSubStreamAvailable:(NSString *)userId available:(bool)available;

/**
 * @brief 某远端用户发布/取消了自己的音频
 *
 * @param userId 远端用户的用户标识
 * @param available 该用户是否发布（或取消发布）了自己的音频，true:
 * 发布；false：取消发布
 */
- (void)onUserAudioAvailable:(NSString *)userId available:(bool)available;

/**
 * @brief 在开启视频自适应场景下，本地流视频根据网络状况自动回退或者恢复
 *
 * @param isFallbackOrRecover true: 本地流视频回退，只推音频
 * false：本地流视频恢复推送
 */
- (void)onLocalVideoFallback:(bool)isFallbackOrRecover;

/**
 * @brief 在开启视频自适应场景下，远端流视频根据网络状况自动回退或者恢复
 *
 * @param userId 远端用户的用户标识
 * @param isFallbackOrRecover true: 远端流视频回退，只拉音频
 * false：远端流视频恢复拉取
 */
- (void)onRemoteVideoFallback:(NSString *)userId
          isFallbackOrRecover:(bool)isFallbackOrRecover;

/**
 * @brief 流控制操作发生错误事件回调
 *
 * @param userId 用户 ID
 * @param streamType 流类型，参见 BRTCCoreVideoStreamType 定义
 * @param controlType 流控制类型，参见 BRTCCoreControlStreamType 定义
 * @param wantMute 是否想要设置为静默
 */
- (void)onControlStreamFailed:(NSString *)userId
                   streamType:(BRTCCoreOCVideoStreamType)streamType
                  controlType:(BRTCCoreOCControlStreamType)controlType
                     wantMute:(BOOL)wantMute;

/**
 * @brief 开始播放远端用户的首帧音频
 *
 * @param userId 远端用户的用户标识
 */
- (void)onFirstAudioFrame:(NSString *)userId;

/**
 * @brief 自己本地的首个视频帧已被发布出去
 *
 * @param streamType 视频流类型，参见 BRTCCoreVideoStreamType 定义
 */
- (void)onSendFirstLocalVideoFrame:(BRTCCoreOCVideoStreamType)streamType;

/**
 * @brief 自己本地的首个音频帧已被发布出去
 *
 */
- (void)onSendFirstLocalAudioFrame;

/**
 * @brief 用户视频大小发生改变回调
 *
 * @param userId 用户标识
 * @param streamType 视频流类型，参见 BRTCCoreVideoStreamType 定义
 * @param newWidth 视频流的宽度
 * @param newHeight 视频流的高度
 */
- (void)onUserVideoSizeChanged:(nullable NSString *)userId
                    streamType:(BRTCCoreOCVideoStreamType)streamType
                      newWidth:(int)newWidth
                     newHeight:(int)newHeight;

/**
 * @brief 音视频技术指标的实时统计回调
 *
 * @param statistics
 * 统计数据，包括自己本地的统计信息和远端用户的统计信息，参见
 * BRTCCoreStatistics 定义
 */
- (void)onStatistics:(BRTCCoreOCStatistics *)statistics;

/**
 * @brief 网速测试的结果回调
 *
 * @param result 网速测试数据数据，包括丢包、往返延迟、上下行的带宽速率，参见
 * BRTCCoreSpeedTestResult 定义
 */
- (void)onSpeedTestResult:(BRTCCoreOCSpeedTestResult *)result;

/**
 * @brief 与服务器的连接已经断开
 *
 */
- (void)onConnectionLost;

/**
 * @brief 正在尝试重新连接到服务器
 *
 */
- (void)onTryToReconnect;

/**
 * @brief 与服务器的连接已经恢复
 *
 */
- (void)onConnectionRecovery;

/**
 * @brief 音量大小的反馈回调
 *
 * @param userVolumes
 * 是一个数组，用于承载所有正在说话的用户的音量大小，取值范围 0 - 100，参见
 * BRTCCoreVolumeInfo 定义
 * @param userVolumesCount 数组大小
 * @param totalVolume 所有远端用户的总音量大小，取值范围 0 - 100
 */
- (void)onUserVoiceVolume:(NSArray<BRTCCoreOCVolumeInfo *> *)userVolumes
              totalVolume:(int)totalVolume;

/**
 * @brief 屏幕分享开启的事件回调
 *
 */
- (void)onScreenCaptureStarted;

/**
 * @brief 屏幕分享暂停的事件回调
 *
 * @param reason 暂停原因：
 * - 0：用户主动暂停
 * - 1：表示设置屏幕分享参数导致的暂停（Windows）
 * - 2：表示屏幕分享窗口被最小化导致的暂停（仅 Windows）
 * - 3：表示屏幕分享窗口被隐藏导致的暂停（仅 Windows）
 */
- (void)onScreenCapturePaused:(int)reason;

/**
 * @brief 屏幕分享恢复的事件回调
 *
 * @param reason 恢复原因：
 * - 0：用户主动恢复
 * - 1：屏幕分享参数设置完毕后自动恢复（Windows）
 * - 2：表示屏幕分享窗口从最小化被恢复（仅 Windows）
 * - 3：表示屏幕分享窗口从隐藏被恢复（仅 Windows）
 */
- (void)onScreenCaptureResumed:(int)reason;

/**
 * @brief 屏幕分享停止的事件回调
 *
 * @param reason 停止原因：
 * - 0：用户主动停止
 * - 1：屏幕窗口关闭导致停止（Windows）
 * - 2：表示屏幕分享的显示屏状态变更
 */
- (void)onScreenCaptureStoped:(int)reason;

/**
 * @brief 本地录制任务已经开始的事件回调
 *
 * @param errCode 错误码 0：初始化录制成功；-1：初始化录制失败；-2:
 * 文件后缀名有误
 * @param storagePath 录制文件存储路径
 */
- (void)onLocalRecordBegin:(int)errCode storagePath:(NSString *)storagePath;

/**
 * @brief 本地录制任务正在进行中的进展事件回调
 *
 * @param duration 已经录制的累计时长，单位毫秒
 * @param storagePath 录制文件存储路径
 */
- (void)onLocalRecording:(long)duration storagePath:(NSString *)storagePath;

/**
 * @brief 本地录制任务已经结束的事件回调
 *
 * @param errCode 错误码
 * 0：录制成功；-1：录制失败；-2：切换分辨率或横竖屏导致录制结束；-3：音频数据或者视频数据一直没有到达导致没有开始正式录制
 * @param storagePath 录制文件存储路径
 */
- (void)onLocalRecordComplete:(int)errCode storagePath:(NSString *)storagePath;

/**
 * @brief 开始向百家云直播 CDN 上发布音视频流的事件回调
 *
 * @param err 0 表示成功，其余值表示失败
 * @param errMsg 具体错误原因
 */
- (void)onStartPublishSelfCDNStream:(int)err errMsg:(NSString *)errMsg;

/**
 * @brief 停止向百家云直播 CDN 上发布音视频流的事件回调
 *
 * @param err 0 表示成功，其余值表示失败
 * @param errMsg 具体错误原因
 */
- (void)onStopPublishSelfCDNStream:(int)err errMsg:(NSString *)errMsg;

/**
 * @brief 开始向非百家云 CDN 上发布音视频流的事件回调
 *
 * @param err 0 表示成功，其余值表示失败
 * @param errMsg 具体错误原因
 */
- (void)onStartPublishOtherCDNStream:(int)err errMsg:(NSString *)errMsg;

/**
 * @brief 停止向非百家云 CDN 上发布音视频流的事件回调
 *
 * @param err 0 表示成功，其余值表示失败
 * @param errMsg 具体错误原因
 */
- (void)onStopPublishOtherCDNStream:(int)err errMsg:(NSString *)errMsg;

/**
 * @brief 回调每个媒体用户的连接质量评价级别
 *
 * @param count 数组中的元素个数
 * @param qualityArray 媒体用户的连接质量评价信息，包含用户ID和级别
 */
- (void)onNetworkQuality:(NSArray<BRTCCoreOCCollectionQuality *> *)qualityArray;

@end

@protocol BRTCCoreOCVTDelegate <NSObject>

/**
 * @brief vt 请求结果事件回调
 *
 * @param resultType vt 请求结果类型，详情见 BRTCCoreVTResultType 定义
 * @param vt_result vt 请求返回结果，json 字符串，只有在 resultType 为
 * BRTCCoreVTResultSuccess 时有效
 * @param errCode vt 请求错误码，只有在 resultType 为 BRTCCoreVTResultFailed
 * 时有效
 * @param errMsg vt 请求错误描述，只有在 resultType 为 BRTCCoreVTResultFailed
 * 时有效
 */
- (void)onVTResult:(BRTCCoreOCVTResult)resultType
          vtResult:(NSString *)vtResult
           errCode:(int)errCode
            errMsg:(NSString *)errMsg;

@end

@protocol BRTCCoreOCVideoRenderDelegate <NSObject>

/**
 * @brief 自定义视频渲染回调
 *
 * @param userId 视频源的 userId
 * @param streamType 视频流类型，参考 BRTCCoreVideoStreamType 定义
 * @param frame 待渲染的视频帧信息，参考 BRTCCoreVideoFrame 定义
 */
- (void)onRenderVideoFrame:(NSString *)userId
                streamType:(BRTCCoreOCVideoStreamType)streamType
                     frame:(BRTCCoreOCVideoFrame *)frame;

@end

@protocol BRTCCoreOCAudioFrameDelegate <NSObject>

/**
 * @brief 本地采集并经过音频模块前处理后的音频数据回调
 *
 * @param frame PCM 格式的音频数据帧，参考 BRTCCoreAudioFrame 定义
 */
- (void)onCapturedRawAudioFrame:(BRTCCoreOCAudioFrame *)frame;

/**
 * @brief 本地采集并经过音频模块前处理、音效处理和混 BGM 后的音频数据回调
 *
 * @param frame PCM 格式的音频数据帧，参考 BRTCCoreAudioFrame 定义
 */
- (void)onLocalProcessedAudioFrame:(BRTCCoreOCAudioFrame *)frame;

/**
 * @brief 混音前的每一路远程用户的音频数据
 *
 * @param frame PCM 格式的音频数据帧，参考 BRTCCoreAudioFrame 定义
 * @param userId 用户标识
 */
- (void)onPlayAudioFrame:(BRTCCoreOCAudioFrame *)frame userId:(NSString *)userId;

/**
 * @brief OnMixedPlayAudioFrame
 * 将各路待播放音频混合之后并在最终提交系统播放之前的数据回调
 *
 * @param frame PCM 格式的音频数据帧，参考 BRTCCoreAudioFrame 定义
 */
- (void)onMixedPlayAudioFrame:(BRTCCoreOCAudioFrame *)frame;

/**
 * @brief 将各路音频混合之后并最终要由系统播放出的音频数据回调用于自定义渲染
 *
 * @param frame PCM 格式的音频数据帧，参考 BRTCCoreAudioFrame 定义
 */
- (void)onCustomAudioRenderingFrame:(BRTCCoreOCAudioFrame *)frame;

@end

@protocol BRTCCoreOCMixExternalAudioProviderDelegate <NSObject>

/**
 * @brief 获取混音数据
 *
 * @param frame 音频数据帧，参考 BRTCCoreAudioFrame 定义
 */
- (void)getMixAudioFrame:(BRTCCoreOCAudioFrame *)frame;

@end

@protocol BRTCCoreOCLogDelegate <NSObject>

/**
 * 日记回调
 *
 * @param log 日志内容
 * @param level 日记等级，参考 BRTCCoreLogLevel 定义
 * @param module 保留字段，暂无具体意义，目前为固定值 BRTCCore
 */
- (void)onLog:(NSString *)log
        level:(BRTCCoreOCLogLevel)level
 moduleString:(NSString *)moduleString;

@end

NS_ASSUME_NONNULL_END

#endif /* BRTCCoreDelegate_h */
