//
//  BRTCCoreOCDef.h
//  BRTCCoreOC
//
//  Created by lw0717 on 2023/4/27.
//

#ifndef BRTCCoreOCDef_h
#define BRTCCoreOCDef_h

#import <Foundation/Foundation.h>

#import "RTCMacros.h"

#define volumeTypeKey @"volume_type"
#define customLogicType @"custom_logic_type"
#define controlCalling @"control_calling"

/**
 * @brief 错误码定义
 *
 */
typedef NS_ENUM(NSInteger, BRTCCoreOCAVError) {
  /// 没有错误
  BRTCCoreOCAVError_Null = 0,
  /// vt 返回结果 json 解析失败
  BRTCCoreOCAVError_Vt_Res_Json_Parse_Fail = -1,
  /// vt 返回结果没有 data 字段
  BRTCCoreOCAVError_Vt_Res_No_Data_Item = -2,
  /// vt 返回结果没有 ut 字段
  BRTCCoreOCAVError_Vt_Res_No_Ut_Item = -3,
  /// vt 返回结果 ut 字段 json 解析失败
  BRTCCoreOCAVError_Vt_Res_Ut_Json_Parse_Fail = -4,
  /// vt 返回结果 payload 为空
  BRTCCoreOCAVError_Vt_Res_Payload_Empty = -5,
  /// vt 返回结果 payload json 解析失败
  BRTCCoreOCAVError_Vt_Res_Payload_Json_Parse_Fail = -6,
  /// Blive 房间配置信息请求返回结果解析为 json 失败
  BRTCCoreOCAVError_Blive_Room_Config_Res_Json_Parse_Fail = -7,
  /// Blive 房间配置信息请求返回结果中没有包含 data 字段
  BRTCCoreOCAVError_Blive_Room_Config_Res_No_Data_Item = -8,
  /// Blive 房间配置信息请求返回结果中没有包含 bt 字段
  BRTCCoreOCAVError_Blive_Room_Config_Res_No_Bt_Item = -9,
  /// Blive 房间配置信息请求返回结果中没有包含 services 字段
  BRTCCoreOCAVError_Blive_Room_Config_Res_No_Services_Item = -10,
  /// Blive 房间配置信息请求返回结果中没有包含 agents 字段
  BRTCCoreOCAVError_Blive_Room_Config_Res_No_Agents_Item = -11,
  /// 获取 Blive 房间配置信息失败，具体失败参考请求回调返回的具体错误码和描述
  BRTCCoreOCAVError_Blive_Room_Config_Res_Fail = -12,
  /// 和 Blive 房间服务器建立连接失败
  BRTCCoreOCAVError_Blive_Room_Config_Connect_Fail = -13,
  /// 登录 Blive 房间服务器失败
  BRTCCoreOCAVError_Blive_Room_Login_Fail = -14,
  /// 请求混流失败，服务器返回了错误通知
  BRTCCoreOCAVError_Blive_Room_Start_Mix_Transcode_Fail = -15,
  /// 请求停止混流失败，服务器返回了错误通知
  BRTCCoreOCAVError_Blive_Room_Stop_Mix_Transcode_Fail = -16,
  /// 打开摄像头失败（windows 使用）
  BRTCCoreOCAVError_Camera_Start_Fail = -1301,
  /// 打开麦克风失败（Windows 使用）
  BRTCCoreOCAVError_Mic_Start_Fail = -1302,
  /// 打开扬声器失败（Windows 使用）
  BRTCCoreOCAVError_Speaker_Start_Fail = -1303,
  /// 打开屏幕采集失败（Windows 使用）
  BRTCCoreOCAVError_Screen_Start_Fail = -1304,
  /// 进房失败，具体失败原因可以在 OnError 中的 -3301 对于的 errMsg 查看
  BRTCCoreOCAVError_Enter_Room_Failed = -3301,
  /// 信令服务器断开连接，无法恢复，可能是网络断开或未知错误，SDK 会自动离房
  BRTCCoreOCAVError_Disconnected = -3302,
};

/**
 * @brief 警告码
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCAVWarning) {
  /// 当前是观众角色，不支持发布音视频，需要先切换成主播角色
  BRTCCoreOCAVWarning_Ignore_Publish_For_Audience = 6001,
};

/**
 * @brief vt 请求结果类型
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCVTResult) {
  /// vt 请求成功，需要解析 vt_result 并发起 EnterRoom
  BRTCCoreOCVTResult_Success = 0,
  /// 请求的 roomId & userId 和上一次一样，忽略
  BRTCCoreOCVTResult_Repeat = 1,
  /// vt 请求失败，向用户通知进房失败
  BRTCCoreOCVTResult_Failed = 2,
};

/**
 * @brief 终端类型定义
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCPlatformType) {
  BRTCCoreOCPlatformType_Mac = 1,
  BRTCCoreOCPlatformType_iOS = 3,
};

/**
 * @brief 角色定义
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCRole) {
  /// 主播
  BRTCCoreOCRole_Anchor = 20,
  /// 观众
  BRTCCoreOCRole_Audience = 21,
};

/**
 * @brief Blive 角色定义
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCBliveRole) {
  /// 观众
  BRTCCoreOCBliveRole_Audience = 0,
  /// 主播
  BRTCCoreOCBliveRole_Anchor = 1,
};

/**
 * @brief 使用场景
 * 不同的使用场景，会有不同的 API接口、回调及行为策略
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCScene) {
  /// 视频通话（实时音视频 RTC）场景
  BRTCCoreOCScene_VideoCall = 0,
  /// 互动直播场景
  BRTCCoreOCScene_Live = 1,
};

/**
 * @brief 房间类型
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCRoomType) {
  /// 正常房间模式，服务器转播所有用户的信令
  BRTCCoreOCRoomType_Normal = 0,
  /// 大房间模式，服务器只转播上麦用户的信令
  BRTCCoreOCRoomType_Big = 1,
};

/**
 * @brief 视频编码器类型
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCCodecType) {
  /// H264 编码器
  BRTCCoreOCCodecType_H264 = 0,
  /// VP8 编码器
  BRTCCoreOCCodecType_VP8 = 1,
};

/**
 * @brief 画质偏好
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCVideoQosPreference) {
  /// 流畅优先
  BRTCCoreOCVideoQosPreference_Smooth = 1,
  /// 清晰优先
  BRTCCoreOCVideoQosPreference_Clear = 2,
};

/**
 * @brief 视频流类型
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCVideoStreamType) {
  /// 主画面视频流
  BRTCCoreOCVideoStreamType_Big = 0,
  /// 小画面视频流
  BRTCCoreOCVideoStreamType_Small = 1,
  /// 辅流（屏幕）
  BRTCCoreOCVideoStreamType_Sub = 2,
};

/**
 * @brief 流控制类型定义
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCControlStreamType) {
    /// 控制音频
    BRTCCoreOCControlStreamTypeAudio = 0,

    /// 控制视频
    BRTCCoreOCControlStreamTypeVideo = 1,
};


/**
 * @brief 视频画面旋转方向
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCVideoRotation) {
  /// 不旋转
  BRTCCoreOCVideoRotation_0 = 0,
  /// 顺时针旋转 90 度
  BRTCCoreOCVideoRotation_90 = 1,
  /// 顺时针旋转 180 度
  BRTCCoreOCVideoRotation_180 = 2,
  /// 顺时针旋转 270 度
  BRTCCoreOCVideoRotation_270 = 3,
};

/**
 * @brief 画面填充模式
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCVideoFillMode) {
  /// 填充模式（铺满）
  BRTCCoreOCVideoFillMode_Fill = 0,
  /// 适应模式（有黑边）
  BRTCCoreOCVideoFillMode_Fit = 1,
};

/**
 * @brief 画面镜像模式
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCVideoMirrorType) {
  /// 开启镜像
  BRTCCoreOCVideoMirrorEnable = 0,
  /// 关闭镜像
  BRTCCoreOCVideoMirrorDisable = 1,
};

/**
 * @brief 水印图片的源类型
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCWaterMarkSrcType) {
  /// 图片文件路径，支持 BMP、GIF、JPEG、PNG、TIFF、Exif、WMF 和 EMF 文件格式
  BRTCCoreOCWaterMarkSrcType_File = 0,
  /// BGRA32 格式内存块
  BRTCCoreOCWaterMarkSrcType_BGRA32 = 1,
  /// RGBA32 格式内存块
  BRTCCoreOCWaterMarkSrcType_RGBA32 = 2,
};

/**
 * @brief 声音音质
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCAudioQuality) {
  /// 人声模式：采样率：16k；单声道；编码码率：16kbps
  BRTCCoreOCAudioQuality_Speech = 0,
  /// 默认模式：采样率：48k；单声道；编码码率：50kbps
  BRTCCoreOCAudioQuality_Default = 1,
  /// 音乐模式：采样率：48k；全频带立体声；编码码率：128kbps
  BRTCCoreOCAudioQuality_Music = 2,
};

/**
 * @brief 视频像素格式
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCVideoPixelFormat) {
  /// YUV420P(I420) 格式
  BRTCCoreOCVideoPixelFormat_I420 = 0,
  /// OpenGL 2D 纹理格式
  BRTCCoreOCVideoPixelFormat_Texture_2D = 1,
  /// BGRA 格式
  BRTCCoreOCVideoPixelFormat_BGRA32 = 2,
  /// NV21 格式
  BRTCCoreOCVideoPixelFormat_NV12 = 3,
  /// RGBA 格式
  BRTCCoreOCVideoPixelFormat_RGBA32 = 4,
};

/**
 * @brief 视频数据传递方式
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCVideoBufferType) {
  /// 使用内存 Buffer 传递视频数据
  BRTCCoreOCVideoBufferType_Buffer = 0,
  /// 使用 Texture 纹理传递视频数据
  BRTCCoreOCVideoBufferType_Texture = 1,
};

/**
 * @brief 视频方向模式
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCOrientationMode) {
  /// 自动模式
  BRTCCoreOCOrientationModeAuto = 0,

  /// 横屏模式
  BRTCCoreOCOrientationModeLandscape = 1,

  /// 竖屏模式
  BRTCCoreOCOrientationModePortrait = 2,
};

/**
 * @brief 日记等级
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCLogLevel) {
  /// 最低级别的 log，将会打印大量信息
  BRTCCoreOCLogLevel_Verbose = 0,
  /// debug 等级
  BRTCCoreOCLogLevel_Info = 1,
  /// 警告日记
  BRTCCoreOCLogLevel_Warning = 2,
  /// 错误日记
  BRTCCoreOCLogLevel_Error = 3,
  /// 不打印日记
  BRTCCoreOCLogLevel_None = 4,
};

/**
 * @brief 日记模块标签
 *
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCLogModule) {
  /// 未知模块
  BRTCCoreOCLogModule_UNKNOWN = 0,

  /// 记录对外接口调用。不仅包括 SDK 调用者的调用，也包括 SDK 内部模块之间
  BRTCCoreOCLogModule_APICALL = 0x01,

  /// 音频设备采集、播放模块，用于音频硬件设备相关的日志
  BRTCCoreOCLogModule_ADM = 0x02,

  /// 视频设备采集模块，用于视频硬件设备相关的日志
  BRTCCoreOCLogModule_VDM = 0x04,

  /// 与信令服务器收、发消息相关的日志
  BRTCCoreOCLogModule_SIGNAL = 0X08,

  /// 上报 action, stats 等信息的数据通道相关的日志
  BRTCCoreOCLogModule_DATACHANNEL = 0x10,

  /// 上报与进房流程相关的日志
  BRTCCoreOCLogModule_JOIN = 0x20,

  /// 与本地流创建、媒体连接相关的日志
  BRTCCoreOCLogModule_LOCAL = 0x40,

  /// 与远端流创建、媒体连接相关的日志
  BRTCCoreOCLogModule_REMOTE = 0x80,

  /// 关键事件发生后，给观察者回调通知的相关日志
  BRTCCoreOCLogModule_EVENT = 0x100,

  /// 主要记录和反馈各种对象（如房间、用户、流）的生命周期相关的日志
  BRTCCoreOCLogModule_OBJECT = 0x200,

  /// 主要记录和反馈编解码相关的日志
  BRTCCoreOCLogModule_CODEC = 0x400,

  /// 主要记录和反馈各种 QOE 相关的日志
  BRTCCoreOCLogModule_QOE = 0x800,

  /// 主要记录和反馈 3A 音频处理相关的日志
  BRTCCoreOCLogModule_APM = 0x1000,
};

/**
 * @brief 日记层级标签
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCStructure) {
  /// 未知层级
  BRTCCoreOCStructure_UNKNOWN = 0,

  /// 特指不同操作系统对应的开发语言层
  BRTCCoreOCStructure_PLATFORM = 1,

  /// 介于平台语言层和跨平台层之间的部分，例如 Android 表示 JNI 层
  BRTCCoreOCStructure_LINK = 2,
};

/**
 * @brief 媒体连接质量评价级别
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCNetworkQualityLevel) {
  BRTCCoreOCNetworkQuality_UNKNOWN = 0,
  BRTCCoreOCNetworkQuality_EXCELLENT,
  BRTCCoreOCNetworkQuality_GOOD,
  BRTCCoreOCNetworkQuality_POOR,
  BRTCCoreOCNetworkQuality_BAD,
  BRTCCoreOCNetworkQuality_VBAD,
  BRTCCoreOCNetworkQuality_DOWN,
};

/**
 * @brief 音频路由
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCAudioRoute) {
  BRTCCoreOCAudioRouteSpeakerphone      = 0, ///< 扬声器，位于手机底部，声音偏大，适合外放音乐。
  BRTCCoreOCAudioRouteEarpiece          = 1, ///< 听筒，位于手机顶部，声音偏小，适合通话。
  BRTCCoreOCAudioRouteWiredHeadset      = 2, ///< 有线耳机
  BRTCCoreOCAudioRouteBluetoothHeadset  = 3, ///< 蓝牙耳机
};

/**
 * @brief 音量类型
 */
typedef NS_ENUM(NSInteger, BRTCCoreOCVolumeType) {
  /// 通话音量
  BRTCCoreOCVolumeTypeVOIP,
  /// 媒体音量
  BRTCCoreOCVolumeTypeMedia,
  /// 媒体音量支持麦克风采集
  BRTCCoreOCVolumeTypeMediaHasMic,
};

/**
 * @brief 音频自定义逻辑类型
 */
typedef NS_ENUM(NSInteger, BRTCCoreOCCustomLogicType) {
  /// 不触发兜底
  BRTCCoreOCCustomLogicTypeeNoUnderTake = 0,
  /// Boom 使用
  BRTCCoreOCCustomLogicTypeBoom = 1,
  /// 音频改造使用(兜底逻辑)
  BRTCCoreOCCustomLogicTypeUnderTake = 2,
};

/**
 * @brief 上报来源。SDK 和 播放器使用的上报 URL 不同，通过 from 区分
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCBliveActionFrom) {
  BRTCCoreOCBliveActionFromSDK,
  BRTCCoreOCBliveActionFromPlayer,
};

/**
 * @brief 上报类型。
 * 1. BRTCCoreOCBliveAction：服务器下发消息及其他不便归类的使用此 action 类型
 * 2. BRTCCoreOCBliveSDKAction：在 SDK 内部执行过程中关键位置上报的 action 类型
 * 3. BRTCCoreOCBliveAPIAction：在对外接口被调用入口处上报此 action 类型
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCBliveActionType) {
  BRTCCoreOCBliveAction,
  BRTCCoreOCBliveSDKAction,
  BRTCCoreOCBliveAPIAction,
};


/**
 * @brief 上报事件名称标识，上报时会将标识转为字符串
 * 如果需要扩展新的上报事件类型，注意全链条都需要补充相关定义
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCBliveActionNameType) {
  BRTCCoreOCBliveActionUndefined,
  BRTCCoreOCBliveActionEnterRoom,
  BRTCCoreOCBliveActionExitRoom,
  BRTCCoreOCBliveActionSwitchRole,
  BRTCCoreOCBliveActionStartMixTranscode,
  BRTCCoreOCBliveActionStopMixTranscode,
  BRTCCoreOCBliveActionUserUpdate,
  BRTCCoreOCBliveActionMixStreamAdded,
  BRTCCoreOCBliveActionMixStreamRemoved,
  BRTCCoreOCBliveActionMixStreamUpdate,
  BRTCCoreOCBliveActionEnterRtcRoom,
  BRTCCoreOCBliveActionExitRtcRoom,
  BRTCCoreOCBliveActionCreatePlayer,
  BRTCCoreOCBliveActionInitPlayer,
  BRTCCoreOCBliveActionStartPlay,
  BRTCCoreOCBliveActionStopPlay,
  BRTCCoreOCBliveActionResumePlay,
  BRTCCoreOCBliveActionNetworkChanged
};

/**
 * @brief 上报结果。
 * 1. BRTCCoreOCBliveActionSuccess：成功
 * 2. BRTCCoreOCBliveActionFail：失败
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCBliveActionResult) {
  BRTCCoreOCBliveActionFail = 0,
  BRTCCoreOCBliveActionSuccess = 1
};

@interface NSString (BRTCCoreOC)

- (instancetype)brtccore_initWithUTF8String:(const char *)nullTerminatedCString;

@end

/**
 * @brief vt 请求参数
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCVTParams : NSObject

/// AppID
@property (nonatomic, copy) NSString *appId;
/// 房间 ID
@property (nonatomic, copy) NSString *roomId;
/// 用户 ID
@property (nonatomic, copy) NSString *userId;
/// 用户传入签名
@property (nonatomic, copy) NSString *userSig;

@end

/**
 * @brief BRTCCore 全局配置参数
 * 该参数用于接口 [Initialize](./BRTCCore.md#Initialize)，注入 BRTCCore
 * 需要的一些全局配置参数
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCGlobalConfig : NSObject

/// 硬编码信息
@property (nonatomic, copy) NSString *hardCodecInfo;
/// opengl 版本号
@property (nonatomic, copy) NSString *openglVersion;
/// 设备型号信息
@property (nonatomic, copy) NSString *deviceInfo;
/// 终端平台类型，详情见 BRTCCoreOCPlatformType 定义
@property (nonatomic, assign) BRTCCoreOCPlatformType platform;
/// 版本号
@property (nonatomic, copy) NSString *version;
///  UA 信息
@property (nonatomic, copy) NSString *userAgent;

@end

//@class RTC_OBJC_TYPE(RTCVideoEncoderFactory);
//@class RTC_OBJC_TYPE(RTCVideoDecoderFactory);

///**
// * @brief BRTCCore 初始化参数
// * 该参数用于接口 Initialize，注入 BRTCCore 需要的一些媒体对象，包括
// * adm，音视频编解码器工厂，webrtc 三大线程等
// *
// */
RTC_OBJC_EXPORT
@interface BRTCCoreOCMediaOption : NSObject
- (instancetype)initConfigDict:(NSDictionary *)configDict;
//@property (nonatomic, strong) RTC_OBJC_TYPE(RTCVideoEncoderFactory) *encoderFactory;
//@property (nonatomic, strong) RTC_OBJC_TYPE(RTCVideoDecoderFactory) *decoderFactory;

@end
//struct BRTCCoreMediaOption {
//  /// 音频设备模块
//  webrtc::AudioDeviceModule* adm;
//
//  /// 音频解码器创建工厂
//  webrtc::AudioDecoderFactory* audioDecoderFactory;
//
//  /// 音频编码器创建工厂
//  webrtc::AudioEncoderFactory* audioEncoderFactory;
//
//  /// 视频解码器创建工厂
//  webrtc::VideoDecoderFactory* videoDecoderFactory;
//
//  /// 视频编码器创建工厂
//  webrtc::VideoEncoderFactory* videoEncoderFactory;
//
//  /// webrtc 信号线程
//  rtc::Thread* webrtcSignalThread;
//
//  /// webrtc 网络线程
//  rtc::Thread* webrtcNetworkThread;
//
//  /// webrtc 工作线程
//  rtc::Thread* webrtcWorkThread;
//
//  BRTCCoreMediaOption()
//      : adm(nullptr),
//        audioDecoderFactory(nullptr),
//        audioEncoderFactory(nullptr),
//        videoDecoderFactory(nullptr),
//        videoEncoderFactory(nullptr),
//        webrtcSignalThread(nullptr),
//        webrtcNetworkThread(nullptr),
//        webrtcWorkThread(nullptr) {}
//};

/**
 * @brief Blive 房间参数
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCBliveRoomParams : NSObject

/// AppID
@property (nonatomic, copy) NSString *appId;
/// 房间 ID
@property (nonatomic, copy) NSString *roomId;
/// 用户 ID
@property (nonatomic, copy) NSString *userId;
/// 用户传入签名
@property (nonatomic, copy) NSString *userSig;
///
@property (nonatomic, assign) BRTCCoreOCBliveRole roleType;
/// 用户 number
@property (nonatomic, copy) NSString *userNumber;
///
@property (nonatomic, copy) NSString *endType;
///
@property (nonatomic, copy) NSString *sdkVersion;
///
@property (nonatomic, copy) NSString *deviceId;
///
@property (nonatomic, copy) NSString *networkType;

@end

/**
 * @brief
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCBliveTask : NSObject

///
@property (nonatomic, copy) NSString *taskId;
///
@property (nonatomic, copy) NSString *customData;

@end

/**
 * @brief
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCBliveTaskList : NSObject

@property (nonatomic, strong) NSArray<BRTCCoreOCBliveTask *> *taskList;

@end

/**
 * @brief
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCBliveRoomConfig : NSObject

///
@property (nonatomic, copy) NSString *reportUrl;
///
@property (nonatomic, copy) NSString *ipUrl;
///
@property (nonatomic, copy) NSString *ip;

@property (nonatomic, assign) NSInteger decodedThreadCount;

@end

/**
 * @brief 
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCBliveMixStreamProperty : NSObject

/// 房间 ID
@property (nonatomic, copy) NSString *roomId;
/// 流 ID
@property (nonatomic, copy) NSString *streamId;
///
@property (nonatomic, copy) NSString *customData;
///
@property (nonatomic, assign) NSInteger timeout;
///
@property (nonatomic, strong) NSArray<NSString *> *streamList;
/// json 字符串
@property (nonatomic, copy) NSString *streamResolutionList;
///
@property (nonatomic, assign) BOOL pull;

@end

/**
 * @brief
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCQueriedBliveMixStream : NSObject

@property (nonatomic, copy) NSString *streamId;

@property (nonatomic, copy) NSString *rtmpUrl;

@property (nonatomic, copy) NSString *httpDnsUrl;

@property (nonatomic, copy) NSString *streamResolutionList;

@end

/**
 * @brief 
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCLayoutsCanvasVideo : NSObject

///
@property (nonatomic, assign) BOOL useQp;
///
@property (nonatomic, copy) NSString *codec;
///
@property (nonatomic, assign) NSInteger bitrate;
///
@property (nonatomic, assign) NSInteger fps;
///
@property (nonatomic, assign) NSInteger gop;

@end

/**
 * @brief 
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCLayoutsCanvas : NSObject

///
@property (nonatomic, assign) NSInteger width;
///
@property (nonatomic, assign) NSInteger height;
///
@property (nonatomic, copy) NSString *color;
///
@property (nonatomic, assign) BOOL isPortrait;
///
@property (nonatomic, assign) NSInteger fitMode;
///
@property (nonatomic, strong) BRTCCoreOCLayoutsCanvasVideo *video;

@end

/**
 * @brief 
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCStreamIdentifierData : NSObject

///
@property (nonatomic, copy) NSString *roomId;
///
@property (nonatomic, copy) NSString *userId;
///
@property (nonatomic, copy) NSString *streamId;

@end

/**
 * @brief 
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCStreamIdentifier : NSObject

///
@property (nonatomic, copy) NSString *type;
///
@property (nonatomic, strong) BRTCCoreOCStreamIdentifierData *data;

@end

/**
 * @brief 
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCCell : NSObject

///
@property (nonatomic, copy) NSString *cellName;
///
@property (nonatomic, strong) BRTCCoreOCStreamIdentifier *streamIdentifier;
///
@property (nonatomic, assign) NSInteger x;
///
@property (nonatomic, assign) NSInteger y;
///
@property (nonatomic, assign) NSInteger width;
///
@property (nonatomic, assign) NSInteger height;
///
@property (nonatomic, assign) NSInteger zOrder;
///
@property (nonatomic, assign) NSInteger fitMode;
///
@property (nonatomic, copy) NSString *color;

@end

/**
 * @brief 
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCMixStreamLayouts : NSObject

///
@property (nonatomic, assign) NSInteger mode;
///
@property (nonatomic, assign) BOOL forceMix;
///
@property (nonatomic, strong) BRTCCoreOCLayoutsCanvas *canvas;
///
@property (nonatomic, strong) NSArray<BRTCCoreOCCell *> *cells;

@end

/**
 * @brief 
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCMixTranscodeParams : NSObject

///
@property (nonatomic, copy) NSString *roomId;
///
@property (nonatomic, copy) NSString *appId;
///
@property (nonatomic, copy) NSString *taskId;
///
@property (nonatomic, copy) NSString *customData;
///
@property (nonatomic, strong) BRTCCoreOCMixStreamLayouts *layouts;

@end

/**
 * @brief 事件上报内容
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCBliveActionReport : NSObject

///
@property (nonatomic, assign) BRTCCoreOCBliveActionFrom from;
///
@property (nonatomic, assign) BRTCCoreOCBliveActionType type;
///
@property (nonatomic, assign) BRTCCoreOCBliveActionNameType name;
///
@property (nonatomic, assign) uint64_t sendTimestamp;
///
@property (nonatomic, assign) uint64_t recvTimestamp;
///
@property (nonatomic, assign) BRTCCoreOCBliveActionResult result;
///
@property (nonatomic, assign) int32_t errCode;
///
@property (nonatomic, copy) NSString *comment;
///
@property (nonatomic, copy) NSString *jsonOption;

@end

/**
 * @brief 进房参数
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCRoomParams : NSObject

/// AppID
@property (nonatomic, copy) NSString *appId;
/// 房间 ID
@property (nonatomic, copy) NSString *roomId;
/// 用户 ID
@property (nonatomic, copy) NSString *userId;
/// 角色，详情见 BRTCCoreOCRole 定义
@property (nonatomic, assign) BRTCCoreOCRole role;
/// 自定义信息，json 格式
@property (nonatomic, copy) NSString *customInfo;
/// webrtc 变量池设置字符串
@property (nonatomic, copy) NSString *fieldTrial;

@end

/**
 * @brief 查询用户 ID 参数
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCQueryId : NSObject

@property (nonatomic, copy) NSString *userId;

@end

/**
 * @brief 房间切换参数
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCSwitchRoomConfig : NSObject

@end


/**
 * @brief 视频编码参数
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCVideoEncParam : NSObject

/// 视频编码器类型，详情见 BRTCCoreOCCodecType
@property (nonatomic, assign) BRTCCoreOCCodecType codecType;
/// 视频目标码率
@property (nonatomic, assign) int videoBitrate;
/// 视频的宽度
@property (nonatomic, assign) int videoWidth;
/// 视频的高度
@property (nonatomic, assign) int videoHeight;
/// 视频的帧率
@property (nonatomic, assign) int videoFps;
/// 视频的方向，详情见 BRTCCoreOCOrientationMode
@property (nonatomic, assign) BRTCCoreOCOrientationMode orientationMode;

@end


/**
 * @brief 网络流控（Qos）参数集
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCNetworkQosParam : NSObject

/// 清晰优先还是流畅优先，详情见 BRTCCoreOCVideoQosPreference
@property (nonatomic, assign) BRTCCoreOCVideoQosPreference preference;
/// 视频流类型, 详情见 BRTCCoreOCVideoStreamType
@property (nonatomic, assign) BRTCCoreOCVideoStreamType streamType;

@end


/**
 * @brief 远端音频流智能并发播放策略的参数
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCAudioParallelParams : NSObject

@end


/**
 * @brief 视频纹理数据
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCTexture : NSObject
/// 视频纹理 ID
@property (nonatomic, assign) int glTextureId;

@end


/**
 * @brief 视频帧信息
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCVideoFrame : NSObject

/// 视频的像素格式，详情见 BRTCCoreVideoPixelFormat
@property (nonatomic, assign) BRTCCoreOCVideoPixelFormat videoFormat;
/// 视频数据结构类型，详情见 BRTCCoreVideoBufferType
@property (nonatomic, assign) BRTCCoreOCVideoBufferType bufferType;
/// bufferType BRTCCoreVideoBufferType_Texture 时的视频数据，承载用于 OpenGL
/// 渲染的纹理数据，详情见 BRTCCoreOCTexture
@property (nonatomic, strong) BRTCCoreOCTexture *texture;
/// bufferType BRTCCoreVideoBufferType_Buffer 时的视频数据
@property (nonatomic, strong) NSData *data;
/// 视频宽度
@property (nonatomic, assign) int width;
/// 视频高度
@property (nonatomic, assign) int height;
/// 视频帧的时间戳，单位毫秒
@property (nonatomic, assign) UInt64 timestamp;
/// 视频像素的顺时针旋转角度
@property (nonatomic, assign) BRTCCoreOCVideoRotation rotation;

@end


/**
 * @brief 音频帧数据
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCAudioFrame : NSObject
/// 音频数据
@property (nonatomic, strong) NSData *data;
/// 采样率
@property (nonatomic, assign) int sampleRate;
/// 声道数
@property (nonatomic, assign) int channel;
/// 时间戳，单位 ms
@property (nonatomic, assign) UInt64 timestamp;

@end


/**
 * @brief 音频自定义回调的格式参数
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCAudioFrameCallbackFormat : NSObject

/// 采样率
@property (nonatomic, assign) int sampleRate;
/// 声道数
@property (nonatomic, assign) int channel;
/// 采样点数
@property (nonatomic, assign) int samplesPerCall;

@end


/**
 * @brief 图片信息
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCImageBuffer : NSObject

/// 图像存储的内容，一般为 BGRA 结构
@property (nonatomic, strong) NSData *buffer;
/// 图像的宽度
@property (nonatomic, assign) int width;
/// 图像的高度
@property (nonatomic, assign) int height;

@end


/**
 * @brief 视频画面的渲染参数
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCRenderParams : NSObject

/// 图像的顺时针旋转角度，详情见 BRTCCoreOCVideoRotation
@property (nonatomic, assign) BRTCCoreOCVideoRotation rotation;
/// 视频画面填充模式，详情见 BRTCCoreOCVideoFillMode
@property (nonatomic, assign) BRTCCoreOCVideoFillMode fillMode;
/// 画面镜像模式, 详情见 BRTCCoreOCVideoMirrorType
@property (nonatomic, assign) BRTCCoreOCVideoMirrorType mirrorType;

@end


/**
 * @brief 本地的音视频统计指标
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCLocalStatistics : NSObject

/// 本地视频的宽度
@property (nonatomic, assign) int width;
/// 本地视频的高度
@property (nonatomic, assign) int height;
/// 本地视频的帧率
@property (nonatomic, assign) int frameRate;
/// 本地视频的码率
@property (nonatomic, assign) int videoBitrate;
/// 本地音频的采样率
@property (nonatomic, assign) int audioSampleRate;
/// 本地音频的码率
@property (nonatomic, assign) int audioBitrate;
/// 视频发送可用带宽（Kbps）
@property (nonatomic, assign) int availableSendBandwidth;
/// 视频流类型
@property (nonatomic, assign) BRTCCoreOCVideoStreamType streamType;
/// 该路音视频流的总丢包率（％）
@property (nonatomic, assign) int finalLoss;

@end


/**
 * @brief 远端的音视频统计指标
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCRemoteStatistics : NSObject

/// 用户 ID
@property (nonatomic, copy) NSString *userId;
/// 该路音视频流的总丢包率（％）
@property (nonatomic, assign) int finalLoss;
/// 音频流的总丢包率（％）
@property (nonatomic, assign) int audioPacketLoss;
/// 该路视频流的总丢包率（％）
@property (nonatomic, assign) int videoPacketLoss;
/// 远端视频的宽度
@property (nonatomic, assign) int width;
/// 远端视频的高度
@property (nonatomic, assign) int height;
/// 远端视频的帧率
@property (nonatomic, assign) int frameRate;
/// 远端视频的码率
@property (nonatomic, assign) int videoBitrate;
/// 远端音频的采样率
@property (nonatomic, assign) int audioSampleRate;
/// 远端音频的码率
@property (nonatomic, assign) int audioBitrate;
/// 视频接收可用带宽（Kbps）
@property (nonatomic, assign) int availableRecvBandwidth;
/// 播放延迟，单位 ms
@property (nonatomic, assign) int jitterBufferDelay;
/// 音频播放的累计卡顿时长，单位 ms
@property (nonatomic, assign) int audioTotalBlockTime;
/// 音频播放卡顿率，单位 (%)
@property (nonatomic, assign) int audioBlockRate;
/// 视频播放的累计卡顿时长，单位 ms
@property (nonatomic, assign) int videoTotalBlockTime;
/// 视频播放卡顿率，单位 (%)
@property (nonatomic, assign) int videoBlockRate;
/// 视频流类型
@property (nonatomic, assign) BRTCCoreOCVideoStreamType streamType;

@end


/**
 * @brief 网络和性能的汇总统计指标
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCStatistics : NSObject

/// 当前应用的 CPU 使用率
@property (nonatomic, assign) int appCpu;
/// 当前系统的 CPU 使用率，单位 (%)
@property (nonatomic, assign) int systemCpu;
/// 从 SDK 到云端的上行丢包率，单位 (%)
@property (nonatomic, assign) int upLoss;
/// 从云端到 SDK 的下行丢包率，单位 (%)
@property (nonatomic, assign) int downLoss;
/// 从 SDK 到云端的往返延时，单位 ms
@property (nonatomic, assign) int rtt;
/// 总发送字节数
@property (nonatomic, assign) UInt64 sentBytes;
/// 总接收字节数;
@property (nonatomic, assign) UInt64 receivedBytes;
/// 本地的音视频统计信息
@property (nonatomic, strong) NSArray<BRTCCoreOCLocalStatistics *> *localStatisticsArray;
/// 远端的音视频统计信息
@property (nonatomic, strong) NSArray<BRTCCoreOCRemoteStatistics *> *remoteStatisticsArray;

@end


/**
 * @brief 音量大小信息
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCVolumeInfo : NSObject

/// 说话者的 userId, 如果 userId 为空则代表是当前用户自己
@property (nonatomic, copy) NSString *userId;
/// 说话者的音量，取值范围 0 - 100
@property (nonatomic, assign) int volume;
/// 是否检测到人声，0：非人声 1：人声。
@property (nonatomic, assign) int vad;

@end


/**
 * @brief 测速参数
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCSpeedTestParams : NSObject
@end


/**
 * @brief 网络测速结果
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCSpeedTestResult : NSObject
@end


/**
 * @brief 本地音频文件的录制参数
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCAudioRecordingParams : NSObject
@end


/**
 * @brief 本地媒体文件的录制参数
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCLocalRecordingParams : NSObject
@end


/**
 * @brief 向非百家云 CDN 上发布音视频流时需设置的转推参数
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCPublishCDNParam : NSObject
@end


/**
 * @brief 设备指标（CPU、内存）提供器，BRTCCore
 * 需要定期通过这个对象获取一些设备指标（平台语言层提供）用于数据上报
 *
 */
@protocol BRTCCoreOCDeviceMetricsInterface <NSObject>

/**
 * @brief 获取应用程序 CPU 占用率
 *
 * @return int CPU 占用率 [0 - 100] <0 代表获取失败
 */
- (int)getAppCpu;

/**
 * @brief 获取应用程序内存适用率
 *
 * @return int 内存适用率 [0 - 100] <0 代表获取失败
 */
- (int)getAppMemUsage;

/**
 * @brief 获取系统整体 CPU 占用率
 *
 * @return int CPU 占用率 [0 - 100] <0 代表获取失败
 */
- (int)getSystemCpu;

/**
 * @brief 获取系统内存适用率
 *
 * @return int 内存适用率 [0 - 100] <0 代表获取失败
 */
- (int)getMemUsage;

/**
 * @brief 获取 wifi 信号强度
 *
 * @return int wifi 信号强度 [0 - 100] 0 代表无信号 100 代表满格信号 <0 代表获取失败
 */
- (int)getWifiSignalStrength;

/**
 * @brief 获取网络连接类型
 *
 * @return const char* 类型描述 例如：wifi、4g、3g、2g、unknown 等
 * 详细见 BRTCCoreOCNetworkType 定义
 */
- (NSString *)getNetworkType;

/**
 * @brief 获取视频渲染统计数据
 *
 * @param userId 用户 ID
 * @param streamType 流类型
 * @return const char* 统计数据，json 格式
 */
- (NSString *)getVideoRenderStates:(NSString *)userId
                        streamType:(BRTCCoreOCVideoStreamType)streamType;

@end

/**
 * @brief 保存每个用户（本地、远端）的媒体连接质量级别
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCCollectionQuality : NSObject

@property (nonatomic, copy) NSString *userId;

@property (nonatomic, assign) BRTCCoreOCNetworkQualityLevel qualityLevel;

@end

#endif /* BRTCCoreDelegate_h */
