//
//  BRTCCoreOCAudioEffectManager.h
//  BRTCCoreOC
//
//  Created by lw0717 on 2023/4/27.
//

#import <Foundation/Foundation.h>

#import "RTCMacros.h"

/**
 * @brief 混响特效
 * 混响特效可以作用于人声之上，通过声学算法对声音进行叠加处理，模拟出各种不同环境下的临场感受
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCVoiceReverbType) {
  /// 关闭特效
  BRTCCoreOCVoiceReverbType_0 = 0,

  /// KTV
  BRTCCoreOCVoiceReverbType_1 = 1,

  /// 小房间
  BRTCCoreOCVoiceReverbType_2 = 2,

  /// 大会堂
  BRTCCoreOCVoiceReverbType_3 = 3,

  /// 低沉
  BRTCCoreOCVoiceReverbType_4 = 4,

  /// 洪亮
  BRTCCoreOCVoiceReverbType_5 = 5,

  /// 金属声
  BRTCCoreOCVoiceReverbType_6 = 6,

  /// 磁性
  BRTCCoreOCVoiceReverbType_7 = 7,

  /// 空灵
  BRTCCoreOCVoiceReverbType_8 = 8,

  /// 录音棚
  BRTCCoreOCVoiceReverbType_9 = 9,

  /// 悠扬
  BRTCCoreOCVoiceReverbType_10 = 10,
};

/**
 * @brief 变声特效
 * 变声特效可以作用于人声之上，通过声学算法对人声进行二次处理，以获得与原始声音所不同的音色
 *
 */
typedef NS_ENUM(NSUInteger, BRTCCoreOCVoiceChangerType) {
  /// 关闭
  BRTCCoreOCVoiceChangerType_0 = 0,

  /// 熊孩子
  BRTCCoreOCVoiceChangerType_1 = 1,

  /// 萝莉
  BRTCCoreOCVoiceChangerType_2 = 2,

  /// 大叔
  BRTCCoreOCVoiceChangerType_3 = 3,

  /// 重金属
  BRTCCoreOCVoiceChangerType_4 = 4,

  /// 感冒
  BRTCCoreOCVoiceChangerType_5 = 5,

  /// 外语腔
  BRTCCoreOCVoiceChangerType_6 = 6,

  /// 困兽
  BRTCCoreOCVoiceChangerType_7 = 7,

  /// 肥宅
  BRTCCoreOCVoiceChangerType_8 = 8,

  /// 强电流
  BRTCCoreOCVoiceChangerType_9 = 9,

  /// 重机械
  BRTCCoreOCVoiceChangerType_10 = 10,

  /// 空灵
  BRTCCoreOCVoiceChangerType_11 = 11,
};

/**
 * @brief 背景音乐的播放控制信息
 * 该信息用于在接口 StartPlayMusic 中指定背景音乐的相关信息，包括播放
 * ID、文件路径和循环次数等 如果使用同一个 ID
 * 播放不同音乐，会先停止播放旧的音乐，再播放新的音乐
 *
 */
RTC_OBJC_EXPORT
@interface BRTCCoreOCAudioMusicParam : NSObject

/// 音乐 ID
/// 【特殊说明】允许播放多路音乐，因此需要使用 ID
/// 进行标记，用于控制音乐的开始、停止、音量等
@property (nonatomic, assign) int id_;

/// 音效文件的完整路径或 URL 地址。支持的音频格式包括 MP3、AAC、M4A、WAV
@property (nonatomic, copy) NSString *path;

/// 音乐循环播放的次数
/// 【推荐取值】取值范围为 0 - 任意正整数，默认值：0。0 表示播放音乐一次；1
/// 表示播放音乐两次；以此类推
@property (nonatomic, assign) int loopCount;

/// 是否将音乐传到远端
/// 【推荐取值】true：音乐在本地播放的同时，远端用户也能听到该音乐；false：主播只能在本地听到该音乐，远端观众听不到。默认值：false
@property (nonatomic, assign) BOOL publish;

/// 播放的是否为短音乐文件
/// 【推荐取值】true：需要重复播放的短音乐文件；false：正常的音乐文件。默认值：false
@property (nonatomic, assign) BOOL isShortFile;

/// 音乐开始播放时间点，单位：毫秒
@property (nonatomic, assign) int startTimeMS;

/// 音乐结束播放时间点，单位毫秒，0 表示播放至文件结尾
@property (nonatomic, assign) int endTimeMS;

@end

/**
 * @brief 背景音乐预加载事件回调
 *
 */
@protocol BRTCCoreOCMusicPreloadDelegate <NSObject>

/**
 * @brief 背景音乐预加载进度
 *
 * @param id 预加载音乐 ID
 * @param progress 预加载进度
 */
- (void)onLoadProgress:(int)id_ progress:(int)progress;

/**
 * @brief 背景音乐预加载出错
 *
 * @param id 预加载音乐 ID
 * @param errorCode
 * -
 * -4001:打开打开文件失败，如音频文件格式不支持，本地音频文件不存在，网络音频文件无法访问等
 * - -4002:解码失败，如音频文件损坏，网络音频文件服务器无法访问等
 * - -4003:预加载数量超上限，请先调用 stopPlayMusic 释放无用的预加载
 */
- (void)onLoadError:(int)id_ errorCode:(int)errorCode;

@end

/**
 * @brief  背景音乐的播放事件回调
 *
 */
@protocol BRTCCoreOCMusicPlayDelegate <NSObject>

/**
 * @brief 背景音乐开始播放
 *
 * @param id 音乐 ID
 * @param errCode 错误码。0: 开始播放成功；-4001:
 * 打开文件失败，如音频文件格式不支持，本地音频文件不存在，网络音频文件无法访问等
 */
- (void)onStart:(int)id_ errCode:(int)errCode;

/**
 * @brief 背景音乐的播放进度
 *
 * @param id 音乐 ID
 * @param curPtsMS 当前播放进度，单位 ms
 * @param durationMS 音乐总长度，单位 ms
 */
- (void)onPlayProgress:(int)id_ curPtsMS:(int)curPtsMS durationMS:(int)durationMS;

/**
 * @brief 背景音乐已经播放完毕
 *
 * @param id 音乐 ID
 * @param errCode 错误码。0: 播放结束；-4002:
 * 解码失败，如音频文件损坏，网络音频文件服务器无法访问等
 */
- (void)onComplete:(int)id_ errCode:(int)errCode;

@end

RTC_OBJC_EXPORT
@interface BRTCCoreOCAudioEffectManager : NSObject

/**
 * @brief 设置预加载事件回调
 *
 * @param observer 具体参考 BRTCCoreMusicPreloadObserver 中定义接口
 */
@property (nonatomic, weak) id<BRTCCoreOCMusicPreloadDelegate> preloadDelegate;

- (instancetype)init NS_UNAVAILABLE;

/**
 * @brief 设置人声的混响效果
 *
 * @param type 混响效果，参考 BRTCCoreVoiceReverbType 中定义
 *
 * @note
 * 设置的效果在退出房间后会自动失效，如果下次进房还需要对应特效，需要调用此接口再次进行设置
 */
- (void)setVoiceReverbType:(BRTCCoreOCVoiceReverbType)type;

/**
 * @brief 设置人声的变声特效
 *
 * @param type 变声特效，参考 BRTCCoreVoiceChangerType 中定义
 *
 * @note
 * 设置的效果在退出房间后会自动失效，如果下次进房还需要对应特效，需要调用此接口再次进行设置
 */
- (void)setVoiceChangerType:(BRTCCoreOCVoiceChangerType)type;

/**
 * @brief 设置语音音量
 *
 * @param volume 音量大小，取值范围为 0 - 100，默认值：100
 */
- (void)setVoiceCaptureVolume:(int)volume;

/**
 * @brief 设置语音音调
 *
 * @param pitch 音调，取值范围为-1.0f~1.0f，默认值：0.0f
 */
- (void)setVoicePitch:(float)pitch;

/**
 * @brief 设置背景音乐的事件回调接口
 *
 * @param musicId 音乐 ID
 * @param observer 回调实例指针，具体参考 BRTCCoreMusicPlayObserver 中定义接口
 */
- (void)setMusicObserver:(int)musicId delegate:(id<BRTCCoreOCMusicPlayDelegate>)delegate;

/**
 * @brief 开始播放背景音乐
 *
 * @param musicParam 音乐参数，具体参考 BRTCCoreAudioMusicParam 中定义
 */
- (void)startPlayMusic:(BRTCCoreOCAudioMusicParam *)musicParam;

/**
 * @brief 停止播放背景音乐
 *
 * @param id 音乐 ID
 */
- (void)stopPlayMusic:(int)musicId;

/**
 * @brief 暂停播放背景音乐
 *
 * @param id 音乐 ID
 */
- (void)pausePlayMusic:(int)musicId;

/**
 * @brief 恢复播放背景音乐
 *
 * @param id 音乐 ID
 */
- (void)resumePlayMusic:(int)musicId;

/**
 * @brief 设置所有背景音乐的本地音量和远端音量的大小
 *
 * @param volume 音量大小，取值范围为 0 - 100，默认值：60
 */
- (void)setAllMusicVolume:(int)volume;

/**
 * @brief 设置某一首背景音乐的远端音量的大小
 *
 * @param id 音乐 ID
 * @param volume 音量大小，取值范围为 0 - 100，默认值：60
 */
- (void)setMusicPublishVolume:(int)musicId volume:(int)volume;

/**
 * @brief 设置某一首背景音乐的本地音量的大小
 *
 * @param id 音乐 ID
 * @param volume 音量大小，取值范围为 0 - 100，默认值：60
 */
- (void)setMusicPlayoutVolume:(int)musicId volume:(int)volume;

/**
 * @brief 获取背景音乐的播放进度（单位：毫秒）
 *
 * @param id 音乐 ID
 * @return long 播放进度
 */
- (long)getMusicCurrentPosInMS:(int)musicId;

/**
 * @brief 获取背景音乐的总时长（单位：毫秒）
 *
 * @param path 音乐文件路径
 * @return long 音乐总时长
 */
- (long)getMusicDurationInMS:(NSString *)path;

/**
 * @brief 设置背景音乐的播放进度（单位：毫秒）
 *
 * @param id 音乐 ID
 * @param pts 要设置的播放进度，单位: 毫秒
 */
- (void)seekMusicToPosInTime:(int)musicId pts:(int)pts;

/**
 * @brief 调整背景音乐的音调高低
 *
 * @param id 音乐 ID
 * @param pitch 音调，默认值是 0.0f，范围是：[-1 ~ 1] 之间的浮点数
 */
- (void)setMusicPitch:(int)musicId pitch:(float)pitch;

/**
 * @brief 调整背景音乐的变速效果
 *
 * @param id 音乐 ID
 * @param speedRate 速度，默认值是 1.0f，范围是：[0.5 ~ 2] 之间的浮点数
 */
- (void)setMusicSpeedRate:(int)musicId speedRate:(float)speedRate;

/**
 * @brief 调整搓碟的变速效果
 *
 * @param id 音乐 ID
 * @param scratchSpeedRate 搓碟速度，默认值是 1.0f，范围是：[-12.0 ~ 12.0]
 * 之间的浮点数, 速度值正/负表示方向正/反，绝对值大小表示速度快慢
 *
 * @note 前置条件 PreloadMusic 成功
 */
- (void)setMusicScratchSpeedRate:(int)musicId scratchSpeedRate:(float)scratchSpeedRate;

/**
 * @brief 设置预加载事件回调
 *
 * @param observer 具体参考 BRTCCoreMusicPreloadObserver 中定义接口
 */
//- (void)setPreloadDelegate:(id<BRTCCoreOCMusicPreloadDelegate>)delegate;

/**
 * @brief 预加载背景音乐
 *
 * @param preloadParam 预加载音乐参数，参考 BRTCCoreAudioMusicParam 中定义
 */
- (void)preloadMusic:(BRTCCoreOCAudioMusicParam *)preloadParam;

@end

