//
//  BRTCCore.h
//  BRTCCoreOC
//
//  Created by lw0717 on 2023/4/26.
//

#import <Foundation/Foundation.h>
#import <UIKit/UIKit.h>

#import "BRTCCoreOCDef.h"
#import "BRTCCoreOCDelegate.h"
#import "BRTCCoreOCVideoRenderer.h"
#import "BRTCCoreOCAudioEffectManager.h"
#import "BRTCCoreOCMediaPlayerManager.h"

#import "RTCMacros.h"

@class RTC_OBJC_TYPE(RTCVideoSource);

NS_ASSUME_NONNULL_BEGIN

RTC_OBJC_EXPORT
@interface BRTCCoreOC : NSObject

/////////////////////////////////////////////////////////////////////////////////
//
//                       初始化相关接口
//
/////////////////////////////////////////////////////////////////////////////////
/**
* @brief 创建 BRTCCore 实例（单例模式）
*
* @return BRTCCore* 指向 BRTCCore 对象的指针
*
* @note 使用 delete BRTCCore* 会导致编译错误，请使用 DestroyBRTCCoreInstance
* 释放对象指针
*/
+ (instancetype)sharedInstance;

/**
* @brief 销毁 BRTCCore 实例（单例模式）
*
* @note CreateBRTCCoreInstance 和 DestroyBRTCCoreInstance
* 确保在同一个线程里调用，避免异步线程操作同一个内存导致异常崩溃
*
*/
+ (void)destroySharedIntance;

/**
 * 设置 BRTCCoreOC 事件回调
 * 设置为 nullptr 表示不接收事件回调，确保在调用 UnInitialize 之前 callback
 * 不被释放，否则会出现崩溃
 *
 * @param callback  回调对象实例指针，详情见 BRTCCoreCallback 定义
 *
 * @note 建议在 CreateBRTCCoreInstance 后马上调用该接口，否则会错过事件回调
 */
@property (nonatomic, weak) id<BRTCCoreOCDelegate> delegate;

/**
 * @brief 设置日志回调
 *
 * @param callback 日志回调，参考 BRTCCoreOCLogDelegate 中定义
 */
@property (nonatomic, weak) id<BRTCCoreOCLogDelegate> logDelegate;

/**
 * 设置设备指标（CPU、内存）提供器
 * 设置为 nullptr 表示取消提供，确保在调用 UnInitialize 之前 provider
 * 不被释放，否则会出现崩溃
 *
 * @param provider 设备指标实例指针，详情见 BRTCCoreDeviceMetricsInterface
 * 定义
 *
 * @note 建议在 CreateBRTCCoreInstance 后马上调用该接口
 */
- (void)setDeviceMetricsProvider:(id<BRTCCoreOCDeviceMetricsInterface>)provider;

/**
 * @brief 发起 vt 请求
 * vt 请求结果通过 OnVTResult 通知
 *
 * @param params vt 请求参数，详情见 BRTCCoreVTParams 定义
 * @param callback vt 请求结果返回的对象指针，详情见 BRTCCoreVTCallback 定义
 *
 * @return 0：成功 <0：失败
 */
- (void)requestVT:(BRTCCoreOCVTParams *)params delegate:(id<BRTCCoreOCVTDelegate>)delegate;

/**
 * @brief 设置一些基础信息，包括平台类型、设备型号、版本号、 BRTCCore
 * 需要的一些媒体对象，包括 adm，音视频编解码器工厂，webrtc 三大线程（移动端）
 *
 * @param config 全局配置参数，详情见 BRTCCoreGlobalConfig 定义
 * @param option 注入对象结构体，详情见 BRTCCoreMediaOption 定义
 * @return true 初始化成功
 * @return false 初始化失败
 *
 * @note 建议在 CreateBRTCCoreInstance
 * 后马上调用该接口，否则会有异常，比如数据通道没法建立等
 */
- (BOOL)initialize:(BRTCCoreOCGlobalConfig *)config option:(BRTCCoreOCMediaOption *)option;

/**
 * @brief 释放所有 BRTCCore 资源，包括
 * SetCallback、SetDeviceMetricsProvider、Initialize 设置的资源
 *
 */
- (void)unInitialize;

/**
 * @brief 创建数据通道
 * 为了方便在 BT 底层使用 BV 的数据通道上报数据，所以单独形成接口
 *
 * @param url 数据通道连接地址
 * @param token 鉴权信息
 *
 * @return 0：成功 <0：失败
 */
- (int)createDataChannel;

/**
 * @brief 销毁数据通道
 *
 */
- (void)destroyDataChannel;

/**
 * @brief 通过数据通道发送数据
 *
 * @param data 待发送数据
 * @param dataSize 待发送数据长度
 *
 * @note 单条数据不要超过 16 KB
 */
- (void)sendDataByDC:(NSData *)data;

/**
 * @brief 通过数据通道发送日记上报
 *
 * @param log 日记数据
 * @param logLevel 日记级别
 * @param module 日记模块，参考 BRTCCoreOCLogModule，可以是多个模块的组合
 * @param structure 日记层级, 参考 BRTCCoreOCStructure
 */
- (void)sendLogReport:(NSString *)log
             logLevel:(BRTCCoreOCLogLevel)logLevel
               moudle:(BRTCCoreOCLogModule)moudle
            structure:(BRTCCoreOCStructure)structure;

/**
 * @brief 调用实验性 API 接口
 *
 * @param jsonStr 接口及参数描述的 JSON 字符串
 */
- (void)callExperimentalAPI:(NSString *)jsonStr;

/////////////////////////////////////////////////////////////////////////////////
//
//                      Blive 相关接口函数
//
/////////////////////////////////////////////////////////////////////////////////

- (int)enterBliveRoom:(BRTCCoreOCBliveRoomParams *)param;

- (void)exitBliveRoom;

- (int)bliveSwitchRole:(BRTCCoreOCBliveRole)role;

- (int)startMixTranscode:(BRTCCoreOCMixTranscodeParams *)param;

- (int)stopMixTranscode:(NSString *)streamId;

/**
 * @brief 查询混流信息
 *
 * @param streamId 混流流ID
 * @param index 混流流索引
 * @param resolution 混流流分辨率
 * @return BRTCCoreQueriedBliveMixStream* 混流信息
 */
- (BRTCCoreOCQueriedBliveMixStream *)queryMixStreamInfo:(NSString *)streamId
                                                  index:(NSInteger)index
                                             resolution:(NSString *)resolution;

- (void)addBliveActionReport:(BRTCCoreOCBliveActionReport *)report;

/////////////////////////////////////////////////////////////////////////////////
//
//                      房间相关接口函数
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * @brief 进入房间
 * 调用该接口后，会收到 OnEnterRoom 回调
 *
 * @param param 进房参数，详情见 BRTCCoreRoomParams 定义
 *
 * @return 0：成功 <0：失败
 *
 * @note 请保证 EnterRoom 与 ExitRoom
 * 前后配对使用，即保证”先退出前一个房间再进入下一个房间”
 */
- (void)enterRoom:(BRTCCoreOCRoomParams *)param;

/**
 * @brief 离开房间
 * 调用该接口用户会离开房间，并且释放摄像头、麦克风、扬声器等资源
 * 等资源释放后，会收到 OnExitRoom 回调，表示离开房间动作完成
 */
- (void)exitRoom;

/**
 * @brief 切换角色
 * 调用本接口可以实现用户在 主播 和 观众 两种角色之间来回切换
 * 可以在进入房间时通过 BRTCCoreRoomParams 中的 role
 * 字段事先确定用户的角色，也可以在进入房间后通过此接口动态切换角色 会收到
 * OnSwitchRole 回调
 *
 * @param role 角色，默认为主播，参考 BRTCCoreRoleType
 */
- (void)switchRole:(BRTCCoreOCRole)role;

/**
 * @brief 设置订阅模式（需要在进房前设置才能生效）
 * 可以通过该接口在“自动订阅”和“手动订阅”两种模式下进行切换：
 * - 自动订阅：默认模式，当收到远端流时会自动订阅，音频会自动播放视频自动解码
 * - 手动订阅：需要手动调用 StartRemoteView
 * 接口才能启动视频流的订阅和解码，需要手动调用 MuteRemoteAudio
 * 接口才能启动声音的播放
 *
 * @param autoRecvAudio 自动订阅音频；false：手动订阅音频。默认值：true
 * @param autoRecvVideo 自动订阅视频；false：手动订阅视频。默认值：true
 */
- (void)setDefaultStreamRecvMode:(BOOL)autoRecvAudio autoRecvVideo:(BOOL)autoRecvVideo;

/**
 * @brief 切换房间
 * 使用该接口可以让用户可以快速从一个房间切换到另一个房间
 * 如果用户的身份是“主播”，该接口在切换房间的同时还会保持自己的音视频发布状态
 * 会收到 OnSwitchRoom 回调
 *
 * @param config 房间参数，详情见 BRTCCoreSwitchRoomConfig 定义
 */
- (void)switchRoom:(BRTCCoreOCSwitchRoomConfig *)config;

/**
 * @brief 请求跨房通话
 * 调用该接口，将另一个房间中某个用户音视频流发布到自己所在的房间中，与此同时，该接口也会将自己的音视频流发布到目标用户的房间中
 * 请求结果会通过 OnConnectOtherRoom 回调
 * 例如：当房间“101”中的用户 A 跟房间“102”中的用户 B 建立跨房通话后：
 * - 房间“101”中的用户都会收到用户 B
 * 的进房和推流事件，即房间“101”中的用户都可以订阅到用户 B 的流
 * - 房间“102”中的用户都会收到用户 A
 * 的进房和推流事件，即房间“102”中的用户都可以订阅到用户 A 的流
 *
 * @param targetRoomid 目标房间 ID
 * @param targetUserid 目标用户 ID
 */
- (void)connectOtherRoom:(NSString *)targetRoomid targetUserid:(NSString *)targetUserid;

/**
 * @brief 退出跨房通话
 * 调用该接口，会断开所有已经通过 ConnectOtherRoom 连接的房间
 * 结果会通过 OnDisconnectOtherRoom 回调
 *
 */
- (void)disconnectOtherRoom;

/**
 * @brief 发送自定义消息给房间内所有用户
 * 房间中的其他用户会收到 OnRecvCustomCmdMsg 回调
 *
 * @param cmdId 消息 ID，取值范围为 1 - 10
 * @param data 待发送的消息，单个消息的最大长度被限制为 1KB
 * @param dataSize 待发送的消息长度
 * @param reliable
 * 是否可靠发送，可靠发送可以获得更高的发送成功率，但可靠发送比不可靠发送会带来更大的接收延迟
 * @param ordered
 * 是否要求有序，即是否要求接收端的数据包顺序和发送端的数据包顺序一致（这会带来一定的接收延时）
 * @return true 消息已经发出
 * @return false 消息发送失败
 *
 * @note
 * - 每秒最多能发送 30 条消息
 * - 每秒最多能发送总计 8KB 数据
 * - 请将 reliable 和 ordered 同时设置为 true 或同时设置为
 * false，暂不支持交叉设置
 */
- (void)sendCustomCmdMsg:(int)cmdId data:(NSData *)data reliable:(BOOL)reliable ordered:(BOOL)ordered;

/**
 * @brief 使用 SEI 将自定义消息附加到视频帧上发送给房间内其它用户
 * 房间中的其他用户会收到 OnRecvSEIMsg 回调
 *
 * @param data 待发送的消息，单个消息的最大长度被限制为 1KB
 * @param dataSize 待发送的消息长度
 * @param repeatCount 重复发送数据次数
 * @return true 消息已通过限制，等待后续视频帧发送
 * @return false 消息被限制发送
 *
 * @note
 * - 每秒最多能发送 30 条消息
 * - 每秒最多能发送总计 8KB 数据
 */
- (void)sendSEIMsg:(NSData *)data repeatCount:(int)repeatCount;

/////////////////////////////////////////////////////////////////////////////////
//
//                       视频相关接口函数
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * @brief 开启本地摄像头的预览画面（仅移动端适用）
 * 在移动端因为采集和渲染是平台相关的，所以需要注入
 * 在 EnterRoom 之前调用此函数，SDK 只会开启摄像头，并一直等到您调用 EnterRoom
 * 之后才开始推视频数据
 * 在 EnterRoom 之后调用此函数，SDK 会开启摄像头并自动开始推视频数据
 *
 * @param videoSource 采集器对象，由平台语言层创建
 * @param videoSink 视频数据 sink 指针，平台语言层用于接收数据并渲染
 *
 * @note 在调用 StopLocalPreview 之后 videoSink 才可以释放，否则会崩溃
 */
- (void)startLocalPreview:(RTC_OBJC_TYPE(RTCVideoSource) *)videoSource
                 renderer:(BRTCCoreOCVideoRenderer *)renderer;

/**
 * @brief 更新本地流的视频源（iOS Android）
 * 不需要重新推流，动态更换视频源
 *
 * @param streamType 要更换视频源的视频流类型（仅支持
 * BRTCCoreVideoStreamTypeBig 和 BRTCCoreVideoStreamTypeSub）
 * @param videoSource 要更换的采集器对象，由平台语言层创建
 */
- (void)updateVideoSource:(BRTCCoreOCVideoStreamType)streamType
              videoSource:(RTC_OBJC_TYPE(RTCVideoSource) *)videoSource;

/**
 * @brief 停止摄像头预览
 * 调用该接口会停止采集，同时取消推送视频数据，同一房间中的其他用户将会收到
 * OnUserVideoAvailable 回调通知
 *
 * @note 对于 Windows 平台，该接口会移除之前通过
 * StartLocalPreview，UpdateLocalView，AddLocalView 接口设置的所有渲染控件
 */
- (void)stopLocalPreview;

/**
 * @brief 暂停/恢复发布本地的视频流
 * 该接口在指定 BRTCCoreVideoStreamTypeBig 时等效于 Start/StopLocalPreview
 * 这两个接口，但具有更好的响应速度
 * - 当暂停/恢复发布指定 BRTCCoreVideoStreamTypeBig
 * 后，同一房间中的其他用户将会收到 OnUserVideoAvailable 回调通知
 * - 当暂停/恢复发布指定 BRTCCoreVideoStreamTypeSub
 * 后，同一房间中的其他用户将会收到 OnUserSubStreamAvailable 回调通知
 *
 * @param streamType 要暂停/恢复的视频流类型（仅支持
 * BRTCCoreVideoStreamTypeBig 和 BRTCCoreVideoStreamTypeSub）
 * @param mute true：暂停；false：恢复
 *
 * @note 当视频和音频都是 mute 状态时，会自动取消发布流
 */
- (void)muteLocalVideo:(BRTCCoreOCVideoStreamType)streamType mute:(BOOL)mute;

/**
 * @brief 设置本地画面被暂停期间的替代图片
 * 当您调用 MuteLocalVideo
 * 暂停本地画面时，您可以通过调用本接口设置一张替代图片，设置后，房间中的其他用户会看到这张替代图片，而不是黑屏画面
 *
 * @param image 设置替代图片，如果传空表示继续传黑帧，默认值为空，参考
 * BRTCCoreImageBuffer
 * @param fps 设置替代图片帧率，最小值为 5，最大值为 10，默认 5
 */
- (void)setVideoMuteImage:(UIImage *)image fps:(int)fps;
//virtual void SetVideoMuteImage(BRTCCoreImageBuffer* image, int fps) = 0;

/**
 * @brief 订阅远端用户的视频流，并绑定视频渲染控件（仅移动端适用）
 * 在移动端因为渲染是平台相关的，所以需要注入 sink 来获取数据
 *
 * @param userId 指定远端用户的 ID
 * @param streamType 指定要观看 userId 的视频流类型，参考
 * BRTCCoreVideoStreamType
 * @param videoSink 接收视频数据对象指针
 *
 * @note
 * - 当指定的 userid 的小画面不存在时，默认切换到该用户的大画面
 * - 在调用 StopRemoteView 之后 videoSink 才可以释放，否则会崩溃
 */
- (void)startRemoteView:(NSString *)userId
             streamType:(BRTCCoreOCVideoStreamType)streamType
               renderer:(BRTCCoreOCVideoRenderer *)renderer;

/**
 * @brief 停止订阅远端用户的视频流，并释放渲染资源
 *
 * @param userId 指定远端用户的 ID
 * @param streamType 要停止的流类型（仅支持 BRTCCoreVideoStreamTypeBig 和
 * BRTCCoreVideoStreamTypeSub）
 *
 * @note 对于 Windows 平台，该接口会移除之前通过
 * StartRemoteView，UpdateRemoteView，AddRemoteView 接口设置的所有渲染控件
 */
- (void)stopRemoteView:(NSString *)userId
            streamType:(BRTCCoreOCVideoStreamType)streamType;

/**
 * @brief 停止订阅所有远端用户的视频流，并释放全部渲染资源
 *
 * @note 如果当前有正在显示的辅路画面（屏幕分享）也会一并被停止
 */
- (void)stopAllRemoteView;

/**
 * @brief 暂停/恢复订阅远端用户的视频流
 * 视频画面会被冻屏在接口调用时的最后一帧
 *
 * @param userId 指定远端用户的 ID
 * @param streamType 要停止的流类型（仅支持 BRTCCoreVideoStreamTypeBig 和
 * BRTCCoreVideoStreamTypeSub）
 * @param mute 是否暂停接收
 *
 * @note 该接口支持您在进入房间前调用，在退出房间之后会被重置
 */
- (void)muteRemoteVideoStream:(NSString *)userId
                   streamType:(BRTCCoreOCVideoStreamType)streamType
                         mute:(BOOL)mute;

/**
 * @brief 暂停/恢复订阅所有远端用户的视频流
 *
 * @param mute 是否暂停接收
 *
 * @note 该接口支持您在进入房间前调用，在退出房间之后会被重置
 */
- (void)muteAllRemoteVideoStreams:(BOOL)mute;

/**
 * @brief 设置视频编码器的编码参数
 *
 * @param param 设置视频编码器的相关参数，详情见 BRTCCoreVideoEncParam 定义
 */
- (void)setVideoEncoderParam:(BRTCCoreOCVideoEncParam *)param;

/**
 * @brief 设置网络质量控制的相关参数
 *
 * @param param 用于设置网络质量控制的相关参数，详情见 BRTCCoreNetworkQosParam
 * 定义
 */
- (void)setNetworkQosParam:(BRTCCoreOCNetworkQosParam *)param;

/**
 * @brief 设置视频编码器输出的画面旋转角度
 * 该设置不影响本地画面的预览方向，但会影响房间中其他用户所观看到的画面方向
 *
 * @param rotation 设置的旋转角度，详情见 BRTCCoreVideoRotation 定义
 */
- (void)setVideoEncoderRotation:(BRTCCoreOCVideoRotation)rotation;

/**
 * @brief 设置视频编码器输出的画面镜像模式
 * 该设置不影响本地画面的镜像模式，但会影响房间中其他用户所观看到的镜像模式
 *
 * @param mirror
 * 是否开启远端镜像，true：开启远端画面镜像；false：关闭远端画面镜像，默认值：false
 */
- (void)setVideoEncoderMirror:(BOOL)mirror;

/**
 * @brief 开启大小画面双路编码模式
 *
 * @param enable 是否开启小画面编码，默认值：false
 * @param smallVideoEncParam 小流的视频编码参数，详情见 BRTCCoreVideoEncParam
 * 定义
 */
- (void)enableSmallVideoStream:(BOOL)enable
            smallVideoEncParam:(BRTCCoreOCVideoEncParam *)smallVideoEncParam;

/**
 * @brief 切换订阅指定远端用户的大小画面
 * 可以通过此接口选定希望订阅的画面是大画面还是小画面，也可以在
 * StartRemoteView 的 streamType 参数指定
 *
 * @param userId 指定远端用户的 ID
 * @param streamType 视频流类型，即选择看大画面
 * BRTCCoreVideoStreamTypeBig 还是小画面
 * BRTCCoreVideoStreamTypeSmall，默认为大画面
 */
- (void)setRemoteVideoStreamType:(NSString *)userId
                      streamType:(BRTCCoreOCVideoStreamType)streamType;

/**
 * @brief 设置弱网条件下流是否开启自适应回退机制，包括本地流和远端流
 * 当开启时，网络不理想的环境下，自动大流关闭保留小流，如果还不行就关闭视频，只留音频
 * 如果网络变好，会从只留音频变成恢复推小流，如果网络还很宽裕，再恢复成推大小流
 *
 * @param enable true: 开启自适应回退机制 false: 禁用自适应回退机制
 *
 * @note 该接口只有在进入房间前调用有效
 */
- (void)setVideoFallbackEnable:(BOOL)enable;

/**
 * @brief 设置 SVC 启动开关
 * SVC 功能是一种视频分层编码技术，可以从时域和空域维度进行控制，目前默认是
 * L1T4，也就是空域 1 层时域 4 层。 启动 SVC
 * 功能可以较好的对抗弱网环境，保证视频流畅播放。 注意：对应的 VT 接口也有控制
 * SVC 启动的开关，当二者冲突时以用户侧设置为准。
 *
 * @param enable YES: 开启 SVC 功能，NO: 禁用 SVC 功能
 * @param streamType 视频流类型，可选择设置主路流 BRTCCoreVideoStreamTypeBig
 * 或者辅流 BRTCCoreVideoStreamTypeSub
 *
 * @note 该接口只有在进入房间前调用有效
 */
- (void)setVideoSvcEnable:(BOOL)enable streamType:(BRTCCoreOCVideoStreamType)streamType;

/**
 * @brief 添加水印
 *
 * @param streamType
 * 要设置水印的流类型（BRTCCoreVideoStreamTypeBig、BRTCCoreVideoStreamTypeSub）
 * @param strData 水印图片源数据（传 nullptr 表示去掉水印）
 * @param srcType 水印图片源数据类型，详情见 BRTCCoreWaterMarkSrcType
 * @param nWidth 水印图片像素宽度（源数据为文件路径时忽略该参数）
 * @param nHeight 水印图片像素高度（源数据为文件路径时忽略该参数）
 * @param xOffset 水印显示的左上角 x 轴偏移，取值范围为 0 - 1 的浮点数
 * @param yOffset 水印显示的左上角 y 轴偏移，取值范围为 0 - 1 的浮点数
 * @param fWidthRatio
 * 水印显示的宽度占画面宽度比例（水印按该参数等比例缩放显示）
 * @param isVisibleOnLocalPreview
 * true：本地预览显示水印；false：本地预览不显示水印
 */
- (void)setWaterMark:(BRTCCoreOCVideoStreamType)streamType
                data:(NSData *)strData
             srcType:(BRTCCoreOCWaterMarkSrcType)srcType
              nWidth:(int)nWidth
             nHeight:(int)nHeight
             xOffset:(float)xOffset
             yOffset:(float)yOffset
         fWidthRatio:(float)fWidthRatio
           isVisible:(bool)isVisibleOnLocalPreview;

/////////////////////////////////////////////////////////////////////////////////
//
//                       音频相关接口函数
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * @brief 开启本地音频的采集和发布
 * 房间中的其他用户会收到 OnUserAudioAvailable 的通知
 *
 * @param quality 声音音质，详情见 BRTCCoreAudioQuality
 */
- (void)startLocalAudio:(BRTCCoreOCAudioQuality)quality;

/**
 * @brief 停止本地音频的采集和发布
 * 房间中的其他用户会收到 OnUserAudioAvailable 的通知
 *
 */
- (void)stopLocalAudio;

/**
 * @brief 暂停/恢复发布本地的音频流
 * 房间中的其他用户会收到 OnUserAudioAvailable 的通知
 * 与 StopLocalAudio 的不同之处在于，会继续发送码率极低的静音包
 *
 * @param mute true：静音；false：恢复
 */
- (void)muteLocalAudio:(BOOL)mute;

/**
 * @brief 暂停/恢复播放指定远端用户的音频流
 * 当您静音某用户的远端音频时，会停止播放指定用户的声音，同时也会停止拉取该用户的音频数据数据
 *
 * @param userId 用于指定远端用户的 ID
 * @param mute true：静音；false：恢复
 *
 * @note 该接口支持您在进入房间前调用，在退出房间之后会被重置
 */
- (void)muteRemoteAudio:(NSString *)userId mute:(BOOL)mute;

/**
 * @brief 暂停/恢复播放所有远端用户的音频流
 *
 * @param mute true：静音；false：恢复
 *
 * @note 该接口支持您在进入房间前调用，在退出房间之后会被重置
 */
- (void)muteAllRemoteAudio:(BOOL)mute;

/**
 * @brief 设置指定远端用户的声音播放音量
 * 操作的音频数据，不改变系统音量
 *
 * @param userId 用于指定远端用户的 ID
 * @param volume 音量大小，取值范围为 0 - 100，默认值：100
 *
 * @note 超过 100 的 volume 会有爆音的风险，请谨慎操作
 */
- (void)setRemoteAudioVolume:(NSString *)userId volume:(int)volume;

/**
 * @brief 设置本地音频的采集音量
 * 操作的音频数据，不改变系统音量
 *
 * @param volume 音量大小，取值范围为 0 - 100，默认值：100
 *
 * @note 超过 100 的 volume 会有爆音的风险，请谨慎操作
 */
- (void)setAudioCaptureVolume:(int)volume;

/**
 * @brief 获取本地音频的采集音量
 *
 * @return int 本地音频的采集音量
 */
- (int)getAudioCaptureVolume;

/**
 * @brief 设置远端音频的播放音量
 * 操作的音频数据，不改变系统音量
 *
 * @param volume 音量大小，取值范围为 0 - 100，默认值：100
 *
 * @note 超过 100 的 volume 会有爆音的风险，请谨慎操作
 */
- (void)setAudioPlayoutVolume:(int)volume;

/**
 * @brief 获取远端音频的播放音量
 *
 * @return int 远端音频的播放音量
 */
- (int)getAudioPlayoutVolume;

/**
 * @brief 启用音量大小提示
 * 开启后，会有 OnUserVoiceVolume 回调通知
 *
 * @param interval 设置 OnUserVoiceVolume 回调的触发间隔，单位为
 * ms，最小间隔为 100ms，如果小于等于 0 则会关闭回调，建议设置为 300ms
 * @param enable_vad true：打开本地人声检测 ；false：关闭本地人声检测
 *
 * @note 如需打开此功能，请在 StartLocalAudio 之前调用才可以生效
 */
- (void)enableAudioVolumeEvaluation:(int)interval enableVad:(BOOL)enableVad;

/**
 * @brief 设置远端音频流智能并发播放策略
 *
 * @param params 音频并发参数，请参见 BRTCCoreAudioParallelParams
 */
- (void)setRemoteAudioParallelParams:(BRTCCoreOCAudioParallelParams *)params;

/**
 * @brief 启用 3D 音效，注意需使用流畅音质 BRTCCoreAudioQualitySpeech
 * 或默认音质 BRTCCoreAudioQualityDefault
 *
 * @param enabled 是否启用 3D 音效，默认为关闭状态
 */
- (void)enable3DSpatialAudioEffect:(BOOL)enabled;

/**
 * @brief 设置 3D 音效中自身坐标及朝向信息
 * 更新自身在世界坐标系中的位置和朝向，会根据该方法参数计算自身和远端用户之间的相对位置，进而渲染出空间音效。注意各参数应分别传入长度为
 * 3 的数组
 *
 * @param position 自身在世界坐标系中的坐标，三个值依次表示前、右、上坐标值
 * @param axisForward
 * 自身坐标系前轴在世界坐标系中的单位向量，三个值依次表示前、右、上坐标值
 * @param axisRight
 * 自身坐标系右轴在世界坐标系中的单位向量，三个值依次表示前、右、上坐标值
 * @param axisUp
 * 自身坐标系上轴在世界坐标系中的单位向量，三个值依次表示前、右、上坐标值
 */
//- (void)updateSelf3DSpatialPosition:()
//virtual void UpdateSelf3DSpatialPosition(int position[3],
//                                         float axisForward[3],
//                                         float axisRight[3],
//                                         float axisUp[3]) = 0;

/**
 * @brief 设置 3D 音效中远端用户坐标信息
 * 更新远端用户在世界坐标系中的位置，会根据自身和远端用户之间的相对位置，进而渲染出空间音效。注意参数为长度等于
 * 3 的数组
 *
 * @param userId 指定远端用户的 ID
 * @param position
 * 该远端用户在世界坐标系中的坐标，三个值依次表示前、右、上坐标值
 */
//- (void)updateRemote3DSpatialPosition:(NSString *)userId;
//virtual void UpdateRemote3DSpatialPosition(const char* userId,
//                                           int position[3]) = 0;

/**
 * @brief 设置指定用户所发出声音的可被接收范围
 * 设置该范围大小之后，该指定用户的声音将在该范围内可被听见，超出该范围将被衰减为
 * 0
 *
 * @param userId 指定远端用户的 ID
 * @param range 声音最大可被接收范围
 */
//- ()
//virtual void Set3DSpatialReceivingRange(const char* userId, int range) = 0;

/**
 * @brief 开启耳返
 * 主播开启耳返后，可以在耳机里听到麦克风采集到的自己发出的声音
 *
 * @param enable true：开启，false：关闭
 */
- (void)enableVoiceEarMonitor:(BOOL)enable;

/**
 * @brief 设置耳返音量
 *
 * @param volume 音量大小，取值范围为 0 - 100，默认值：100
 */
- (void)setVoiceEarMonitorVolume:(int)volume;

/**
 * @brief 开启或关闭播放一个空数据的音频源
 * 可以使用这个接口控制音频设备的生命周期
 *
 * @param enable YES：打开 NO：关闭
 */
- (void)enableFakeAudioSourcePlay:(BOOL)enable;

/////////////////////////////////////////////////////////////////////////////////
//
//                      屏幕分享相关接口函数
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * @brief 开始桌面端屏幕分享（仅移动端适用）
 * 移动端的屏幕采集是平台相关的，所以需要通过注入采集器对象的方式
 *
 * @param streamType
 * 屏幕分享使用的线路，可以设置为主路（BRTCCoreOCVideoStreamTypeBig）或者辅路（BRTCCoreOCVideoStreamTypeBub），推荐使用辅路
 * @param encParam
 * 屏幕分享的画面编码参数，如果不为空，优先使用此参数设置的编码参数，参考
 * BRTCCoreVideoEncParam
 * - 如果 encParam 为空，但是已经调用过 SetSubStreamEncoderParam
 * 设置辅路视频编码参数，将使用设置过的辅路编码参数进行屏幕分享
 * - 如果 encParam 为空，而且未调用过 SetSubStreamEncoderParam
 * 设置辅路视频编码参数，将默认使用采集原始宽高作为编码参数
 * @param videoSource 屏幕采集器对象指针
 * @param renderer 屏幕采集数据 sink 对象指针
 *
 * @note 在调用 StopScreenCapture 之后 videoSource 才可以释放，否则会崩溃
 * @note
 * 默认情况下，屏幕分享使用辅路画面。如果使用主路做屏幕分享，您需要提前停止摄像头采集（StopLocalPreview）以避免相互冲突
 */
- (void)startScreenCapture:(BRTCCoreOCVideoStreamType)streamType
             videoEncParam:(BRTCCoreOCVideoEncParam *)encParam
               videoSource:(RTC_OBJC_TYPE(RTCVideoSource) *)videoSource
                  renderer:(BRTCCoreOCVideoRenderer *)renderer;

/**
 * @brief 停止屏幕分享
 *
 */
- (void)stopScreenCapture;

/**
 * 暂停屏幕分享
 */
- (void)pauseScreenCapture;

/**
 * 恢复屏幕分享
 */
- (void)resumeScreenCapture;

/**
 * @brief 设置屏幕分享（即辅路）的视频编码参数
 * 请注意如下两个接口的差异：
 * - SetVideoEncoderParam
 * 用于设置主路画面（BRTCCoreVideoStreamTypeBig，一般用于摄像头）的视频编码参数
 * - SetSubStreamEncoderParam
 * 用于设置辅路画面（BRTCCoreVideoStreamTypeSub，一般用于屏幕分享）的视频编码参数
 *
 * @param param 辅流编码参数，详情见 BRTCCoreVideoEncParam 定义
 *
 * @note 即使您使用主路传输屏幕分享（在调用 StartScreenCapture 时设置 type =
 * BRTCCoreVideoStreamTypeBig），依然要使用 SetSubStreamEncoderParam
 * 设定屏幕分享的编码参数，而不要使用 SetVideoEncoderParam
 */
- (void)setSubStreamEncoderParam:(BRTCCoreOCVideoEncParam *)param;

/////////////////////////////////////////////////////////////////////////////////
//
//                      自定义采集和渲染
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * @brief 启用/关闭视频自定义采集模式
 * 开启该模式后，不在运行原有的视频采集流程，即不再继续从摄像头采集数据和美颜，而是只保留视频编码和发送能力
 * 需要通过 SendCustomVideoData 不断地塞入自己采集的视频画面
 *
 * @param streamType 用于指定视频流类型，仅支持 BRTCCoreVideoStreamTypeBig 和
 * BRTCCoreVideoStreamTypeSub
 * @param enable 是否启用，默认值：false
 */
- (void)enableCustomVideoCapture:(BRTCCoreOCVideoStreamType)streamType
                          enable:(BOOL)enable
                     videoSource:(RTC_OBJC_TYPE(RTCVideoSource) *)videoSource;

/**
 * @brief 投送外部采集的视频帧
 *
 * @param streamType 用于指定视频流类型，仅支持 BRTCCoreVideoStreamTypeBig 和
 * BRTCCoreVideoStreamTypeSub
 * @param frame 视频数据，详情见 BRTCCoreVideoFrame 定义
 */
- (void)sendCustomVideoData:(BRTCCoreOCVideoStreamType)streamType
                      frame:(BRTCCoreOCVideoFrame *)frame;

/**
 * @brief 启用/关闭音频自定义采集模式
 * 开启该模式后，不在运行原有的音频采集流程，即不再继续从麦克风采集音频数据，而是只保留音频编码和发送能力
 * 需要通过 SendCustomAudioData 不断地塞入自己采集的音频数据
 *
 * @param enable 是否启用，默认值：false
 */
- (void)enableCustomAudioCapture:(BOOL)enable;

/**
 * @brief 投送外部采集的音频数据
 *
 * @param frame 音频数据，详情见 BRTCCoreAudioFrame 定义
 */
- (void)sendCustomAudioData:(BRTCCoreOCAudioFrame *)frame;

/**
 * @brief 启用/关闭自定义音轨
 *
 * @param enablePublish 控制混入的音轨是否要在远端播放，默认值：false
 * @param enablePlayout 控制混入的音轨是否要在本地播放，默认值：false
 */
- (void)enableMixExternalAudioFrame:(BOOL)enablePublish enablePlayout:(BOOL)enablePlayout;

/**
 * @brief 设置自定义音轨数据提供器
 *
 * @param provider 自定义音轨数据提供器
 *
 * @note 该方法不能和 MixExternalAudioFrame
 * 同时使用，如果调用了该方法，MixExternalAudioFrame 将不会生效
 */
- (void)setMixExternalAudioProviderDelegate:(id<BRTCCoreOCMixExternalAudioProviderDelegate>)delegate;

/**
 * @brief 混入自定义音轨
 *
 * @param frame 音频数据，详情见 BRTCCoreAudioFrame 定义
 * @return int 缓冲的长度，单位：ms。< 0 错误
 */
- (void)mixExternalAudioFrame:(BRTCCoreOCAudioFrame *)frame;

/**
 * @brief 设置推流时混入外部音频的推流音量和播放音量
 *
 * @param publishVolume 设置的推流音量大小，范围 0 - 100，-1 表示不改变
 * @param playoutVolume 设置的播放音量大小，范围 0 - 100，-1 表示不改变
 */
- (void)setMixExternalAudioVolume:(int)publishVolume playoutVolume:(int)playoutVolume;

/**
 * @brief 生成自定义采集时的时间戳
 *
 * @return UInt64 调用本接口当时的 PTS 时间戳
 */
- (UInt64)generateCustomPTS;

/**
 * @brief 设置本地视频自定义渲染回调
 * 设置该回调之后会跳过原来的渲染流程，并把采集到的数据回调出来
 * 通过设置回调为 nullptr 停止回调
 *
 * @param pixelFormat 指定回调的像素格式，参考 BRTCCoreVideoPixelFormat
 * @param bufferType 指定视频数据结构类型，目前只支持
 * BRTCCoreVideoBufferType_Buffer
 * @param delegate 自定义渲染回调对象指针，详情见 BRTCCoreOCVideoRenderDelegate
 * @return int 0：成功；<0：错误
 */
- (int)setLocalVideoRenderDelegate:(id<BRTCCoreOCVideoRenderDelegate>)delegate
                       pixelFormat:(BRTCCoreOCVideoPixelFormat)pixelFormat
                        bufferType:(BRTCCoreOCVideoBufferType)bufferType;

/**
 * @brief 设置远端视频自定义渲染回调
 * 设置该回调之后会跳过原来的渲染流程，并把采集到的数据回调出来
 * 通过设置回调为 nullptr 停止回调
 *
 * @param userId 远端用户 ID
 * @param pixelFormat 指定回调的像素格式，参考 BRTCCoreVideoPixelFormat
 * @param bufferType 指定视频数据结构类型，目前只支持
 * BRTCCoreVideoBufferType_Buffer
 * @param callback 自定义渲染回调对象指针，详情见 BRTCCoreVideoRenderCallback
 * @return int 0：成功；<0：错误
 */
- (int)setRemoteVideoRenderDelegate:(NSString *)userId
                           delegate:(id<BRTCCoreOCVideoRenderDelegate>)delegate
                        pixelFormat:(BRTCCoreOCVideoPixelFormat)pixelFormat
                         bufferType:(BRTCCoreOCVideoBufferType)bufferType;

/**
 * @brief 设置音频数据自定义回调
 * 设置该回调之后，内部会把音频数据（PCM 格式）回调出来，包括：
 * - OnCapturedRawAudioFrame：本地麦克风采集到的原始音频数据回调
 * - OnLocalProcessedAudioFrame：本地采集并经过音频模块前处理后的音频数据回调
 * - OnRemoteUserAudioFrame：混音前的每一路远程用户的音频数据
 * -
 * OnMixedPlayAudioFrame：将各路音频混合之后并最终要由系统播放出的音频数据回调
 * -
 * OnCustomAudioRenderingFrame：将各路音频混合之后并最终要由系统播放出的音频数据回调用于自定义渲染，内部会跳过渲染流程，需要调用
 * EnableCustomAudioRendering 开启音频自定义渲染才生效。 区别于
 * OnMixedPlayAudioFrame
 *
 * @param callback 音频数据回调对象指针，详情见 BRTCCoreAudioFrameCallback
 * 定义
 * @return int 0：成功；<0：错误
 *
 * @note
 * 设置回调为空即代表停止自定义音频回调，反之，设置回调不为空则代表启动自定义音频回调
 */
- (int)setAudioFrameDelegate:(id<BRTCCoreOCAudioFrameDelegate>)delegate;

/**
 * @brief 设置本地麦克风采集出的原始音频帧回调格式
 * 本接口用于设置 OnCapturedRawAudioFrame 回调出来的 AudioFrame 的格式
 *
 * @param format 音频数据回调格式，详情见 BRTCCoreAudioFrameCallbackFormat
 * 定义
 * @return int 0：成功；<0：错误
 */
- (int)setCapturedRawAudioFrameCallbackFormat:(BRTCCoreOCAudioFrameCallbackFormat *)format;

/**
 * @brief 设置经过前处理后的本地音频帧回调格式
 * 本接口用于设置 OnLocalProcessedAudioFrame 回调出来的 AudioFrame 的格式
 *
 * @param format 音频数据回调格式，详情见 BRTCCoreAudioFrameCallbackFormat
 * 定义
 * @return int 0：成功；<0：错误
 */
- (int)setLocalProcessedAudioFrameCallbackFormat:(BRTCCoreOCAudioFrameCallbackFormat *)format;

/**
 * @brief 设置最终要由系统播放出的音频帧回调格式
 * 本接口用于设置 OnMixedPlayAudioFrame 回调出来的 AudioFrame 的格式
 *
 * @param format 音频数据回调格式，详情见 BRTCCoreAudioFrameCallbackFormat
 * 定义
 * @return int 0：成功；<0：错误
 */
- (void)setMixedPlayAudioFrameCallbackFormat:(BRTCCoreOCAudioFrameCallbackFormat *)format;

/**
 * @brief 设置自定义音频渲染的音频帧回调格式
 * 本接口用于设置 OnCustomAudioRenderingFrame 回调出来的 AudioFrame 的格式
 *
 * @param format 音频数据回调格式，详情见 BRTCCoreAudioFrameCallbackFormat
 * 定义
 * @return int 0：成功；<0：错误
 */
- (int)setCustomAudioRenderingFrameCallbackFormat:(BRTCCoreOCAudioFrameCallbackFormat *)format;

/**
 * @brief 开启音频自定义播放
 * 启用音频自定义播放后，内部将不再调用系统的音频接口播放数据，音频数据会通过
 * OnCustomAudioRenderingFrame 回调
 *
 * @param enable 是否启用音频自定义播放，默认为关闭状态
 *
 * @note 需要您在进入房间前设置才能生效，暂不支持进入房间后再设置
 */
- (void)enableCustomAudioRendering:(BOOL)enable;

/////////////////////////////////////////////////////////////////////////////////
//
//                      网络测试相关接口
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * @brief 开始进行网速测试（进入房间前使用）
 *
 * @param params 测速选项，参考 BRTCCoreSpeedTestParams 中定义
 * @return int 0：成功，< 0：失败
 *
 * @note 请在进入房间前进行网速测试
 * @note 同一时间只允许一项网速测试任务运行
 */
- (int)startSpeedTest:(BRTCCoreOCSpeedTestParams *)params;

/**
 * @brief 停止网络测速
 *
 */
- (void)stopSpeedTest;

/////////////////////////////////////////////////////////////////////////////////
//
//                      本地录制相关接口
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * @brief 开始录音
 *
 * @param param 录音参数，参考 BRTCCoreAudioRecordingParams 中定义
 * @return int
 * 0：成功；-1：录音已开始；-2：文件或目录创建失败；-3：后缀指定的音频格式不支持
 */
- (int)startAudioRecording:(BRTCCoreOCAudioRecordingParams *)param;

/**
 * @brief 停止录音
 *
 */
- (void)stopAudioRecording;

/**
 * @brief 开始本地媒体录制
 *
 * @param params 录制参数，参考 BRTCCoreLocalRecordingParams 中定义
 */
- (void)startLocalRecording:(BRTCCoreOCLocalRecordingParams *)params;

/**
 * @brief 停止本地媒体录制
 *
 */
- (void)stopLocalRecording;

/////////////////////////////////////////////////////////////////////////////////
//
//                      CDN 相关接口
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * @brief 开始向百家云直播 CDN 上发布音视频流
 *
 * @param streamId 自定义流 ID
 * @param streamType 流类型，仅支持 BRTCCoreVideoStreamTypeBig 和
 * BRTCCoreVideoStreamTypeSub
 */
- (void)startPublishSelfCDNStream:(NSString *)streamId streamType:(BRTCCoreOCVideoStreamType)streamType;

/**
 * @brief 停止向百家云直播 CDN 上发布音视频流
 *
 */
- (void)stopPublishSelfCDNStream;

/**
 * @brief 开始向非百家云 CDN 上发布音视频流
 *
 * @param param CDN 转推参数，参考 BRTCCorePublishCDNParam 中定义
 */
- (void)startPublishOtherCDNStream:(BRTCCoreOCPublishCDNParam *)param;

/**
 * @brief 停止向非百家云 CDN 上发布音视频流
 *
 */
- (void)stopPublishOtherCDNStream;

/////////////////////////////////////////////////////////////////////////////////
//
//                      其它
//
/////////////////////////////////////////////////////////////////////////////////

/**
 * @brief 获取音效管理类实例
 *
 * @return BRTCCoreOCAudioEffectManager 对象，参考
 * BRTCCoreOCAudioEffectManager
 */
- (BRTCCoreOCAudioEffectManager *)getAudioEffectManager;

/**
 * @brief 获取媒体播放管理类实例
 * 
 */
- (BRTCCoreOCMediaPlayerManager *)getMediaPlayerManager;

/**
 * @brief 设置日记等级
 *
 * @param level 日记等级，详情见 BRTCCoreLogLevel
 */
- (void)setLogLevel:(BRTCCoreOCLogLevel)level;

@end

NS_ASSUME_NONNULL_END
