//
//  BRTCCoreAdapter+MediaPlayerManager.h
//  BRTC-i
//
//  Created by lw0717 on 2024/9/2.
//  Copyright © 2024 BRTC. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "BRTCCoreAdapter.h"
#import "BRTCCoreAdapter+Tools.h"

#import <BRTCCore/BRTCCoreOCMediaPlayerManager.h>

NS_ASSUME_NONNULL_BEGIN

@interface BRTCCoreAdapter (MediaPlayerManager) <BRTCCoreOCMediaPlayerDelegate>

/**
 * 设置播放状态回调接口
 *
 * @param mediaPlayerDelegate  具体参考 BRTCMediaPlayerDelegate 中定义接口
 */
- (void)setupMediaPlayerDelegate:(id<BRTCMediaPlayerDelegate>)mediaPlayerDelegate;

/**
 * 设置视频显示窗口
 *
 * @param view 承载视频画面的控件
 *
 * @note 在调用 Stop 之后 videoSink 才可以释放，否则
 * Play 会崩溃
 *
 * @return  0: 设置成功
 *         <0: 设置失败
 */
- (int)setVideoRenderer:(int)playerId
                   view:(BRTCView *)view;

/**
 * 打开媒体资源
 *
 * @param source 媒体资源参数
 *
 * @return  0: 设置成功
 *         <0: 设置失败
 */
- (int)open:(BRTCMediaPlayerSource *)source;

/**
 * 开始播放
 *
 * @note 必须等待 open 成功后才可以调用
 *
 * @param playerId  播放器 ID
 *
 * @return  0: 播放成功
 *         <0: 播放失败
 */
- (int)play:(int)playerId;

/**
 * 停止播放
 *
 * @note 停止播放后，如需重新播放，需要重新调用 open, 非 autoPlay
 * 的，还需要再调用 play 开始播放
 *
 * @param playerId  播放器 ID
 *
 * @return  0: 停止成功
 *         <0: 停止失败
 */
- (int)stop:(int)playerId;

/**
 * 定位到媒体文件的指定播放位置（单位：毫秒）
 *
 * @note 如果在播放过程中调用该接口，SDK 会自动从指定位置开始播放
 *       如果在暂停状态下调用该接口，SDK
 * 会定位到您指定的位置，如需播放，请调用 resume
 *
 * @param playerId  播放器 ID
 * @param newPos 指定的播放位置 单位毫秒
 *
 * @return  0: 设置成功
 *         <0: 设置失败
 */
- (int)seek:(int)playerId newPos:(long)newPos;

/**
 * 暂停播放
 *
 * @param playerId  播放器 ID
 *
 * @return  0: 暂停成功
 *         <0: 暂停失败
 */
- (int)pause:(int)playerId;

/**
 * 恢复播放
 *
 * @param playerId  播放器 ID
 *
 * @return  0: 恢复成功
 *         <0: 恢复失败
 */
- (int)resume:(int)playerId;

/**
 * 设置是否静音
 *
 * @param playerId  播放器 ID
 * @param muted true: 静音  false: 取消静音
 *
 * @return  0: 设置成功
 *         <0: 设置失败
 */
- (int)muteAudio:(int)playerId muted:(BOOL)muted;

/**
 * 同时设置该媒体中音频的本地音量和远端音量的大小
 *
 * - 本地音量：即主播本地可以听到的播放音量大小。
 * - 远端音量：即观众端可以听到的音频的音量大小。
 *
 * @param playerId  播放器 ID
 * @param volume 音量大小，取值范围为0 - 100，默认值：100。
 *
 * @note 如果将 volume 设置成 100 之后感觉音量还是太小，可以将 volume
 * 最大设置成 150，但超过 100 的 volume 会有爆音的风险，请谨慎操作。
 *
 * @return  0: 设置成功
 *         <0: 设置失败
 */
- (int)setAllAudioVolume:(int)playerId volume:(int)volume;

/**
 * 设置播放器中音频在远端听到的音量的大小
 *
 * - 观众端可听到的音频的音量大小。
 *
 * @param playerId  播放器 ID
 * @param volume 音量大小，取值范围为0 - 100；默认值：100
 *
 * @note 如果将 volume 设置成 100 之后感觉音量还是太小，可以将 volume
 * 最大设置成 150，但超过 100 的 volume 会有爆音的风险，请谨慎操作。
 * @return  0: 设置成功
 *         <0: 设置失败
 */
- (int)setAudioPublishVolume:(int)playerId volume:(int)volume;

/**
 * 设置播放器中音频在本地播放音量的大小
 *
 * @param playerId  播放器 ID
 * @param volume 音量大小，取值范围为0 - 100，默认值：100。
 *
 * @note 如果将 volume 设置成 100 之后感觉音量还是太小，可以将 volume
 * 最大设置成 150，但超过 100 的 volume 会有爆音的风险，请谨慎操作。
 *
 * @return 0: 设置成功
 *        <0: 设置失败
 */
- (int)setAudioPlayoutVolume:(int)playerId volume:(int)volume;

/**
 * 获取当前播放进度（单位：毫秒）
 *
 * @param playerId  播放器 ID
 *
 * @return 成功返回当前播放时间，单位：毫秒，失败返回-1
 */
- (long)getCurrentPlayPosInMS:(int)playerId;

/**
 * 获取媒体文件总时长（单位：毫秒）
 *
 * @param playerId  播放器 ID
 *
 * @return 成功返回时长，失败返回-1
 */
- (long)getDurationInMS:(int)playerId;

@end

NS_ASSUME_NONNULL_END
