//
// BrtcEnumerates.h
// BrtcEngineKit
//
// Copyright (c) 2018 Brtc. All rights reserved.
//

#import <Foundation/Foundation.h>

/** Warning code.

Warning codes occur when the SDK encounters an error that may be recovered automatically. These are only notifications, and can generally be ignored. For example, when the SDK loses connection to the server, the SDK reports the BrtcWarningCodeOpenChannelTimeout(106) warning and tries to reconnect automatically.
*/
typedef NS_ENUM(NSInteger, BrtcWarningCode) {
    /** 8: The specified view is invalid. Specify a view when using the video call function. */
    BrtcWarningCodeInvalidView = 8,
    /** 16: Failed to initialize the video function, possibly caused by a lack of resources. The users cannot see the video while the voice communication is not affected. */
    BrtcWarningCodeInitVideo = 16,
     /** 20: The request is pending, usually due to some module not being ready, and the SDK postpones processing the request. */
    BrtcWarningCodePending = 20,
    /** 103: No channel resources are available. Maybe because the server cannot allocate any channel resource. */
    BrtcWarningCodeNoAvailableChannel = 103,
    /** 104: A timeout occurs when looking up the channel. When joining a channel, the SDK looks up the specified channel. The warning usually occurs when the network condition is too poor for the SDK to connect to the server. */
    BrtcWarningCodeLookupChannelTimeout = 104,
    /** 105: The server rejects the request to look up the channel. The server cannot process this request or the request is illegal.
     <br></br><b>DEPRECATED</b> as of v2.4.1. Use BrtcConnectionChangedRejectedByServer(10) in the `reason` parameter of [connectionChangedToState]([BrtcEngineDelegate rtcEngine:connectionChangedToState:reason:]).
     */
    BrtcWarningCodeLookupChannelRejected = 105,
    /** 106: The server rejects the request to look up the channel. The server cannot process this request or the request is illegal. */
    BrtcWarningCodeOpenChannelTimeout = 106,
    /** 107: The server rejects the request to open the channel. The server cannot process this request or the request is illegal. */
    BrtcWarningCodeOpenChannelRejected = 107,
    /** 111: A timeout occurs when switching to the live video. */
    BrtcWarningCodeSwitchLiveVideoTimeout = 111,
    /** 118: A timeout occurs when setting the client role in the live broadcast profile. */
    BrtcWarningCodeSetClientRoleTimeout = 118,
    /** 119: The client role is unauthorized. */
    BrtcWarningCodeSetClientRoleNotAuthorized = 119,
    /** 121: The ticket to open the channel is invalid. */
    BrtcWarningCodeOpenChannelInvalidTicket = 121,
    /** 122: Try connecting to another server. */
    BrtcWarningCodeOpenChannelTryNextVos = 122,
    /** 701: An error occurs in opening the audio mixing file. */
    BrtcWarningCodeAudioMixingOpenError = 701,
    /** 1014: Audio Device Module: a warning occurs in the playback device. */
    BrtcWarningCodeAdmRuntimePlayoutWarning = 1014,
    /** 1016: Audio Device Module: a warning occurs in the recording device. */
    BrtcWarningCodeAdmRuntimeRecordingWarning = 1016,
    /** 1019: Audio Device Module: no valid audio data is collected. */
    BrtcWarningCodeAdmRecordAudioSilence = 1019,
    /** 1020: Audio Device Module: a playback device fails. */
    BrtcWarningCodeAdmPlaybackMalfunction = 1020,
    /** 1021: Audio Device Module: a recording device fails. */
    BrtcWarningCodeAdmRecordMalfunction = 1021,
    /** 1025: The system interrupts the audio session, and the session is no longer active. */
    BrtcWarningCodeAdmInterruption = 1025,
    /** 1031: Audio Device Module: the recorded audio is too low. */
    BrtcWarningCodeAdmRecordAudioLowlevel = 1031,
    /** 1032: Audio Device Module: the playback audio is too low. */
    BrtcWarningCodeAdmPlayoutAudioLowlevel = 1032,
    /** 1051: Audio Device Module: howling is detected. */
    BrtcWarningCodeApmHowling = 1051,
    /** 1052: Audio Device Module: the device is in the glitch state. */
    BrtcWarningCodeAdmGlitchState = 1052,
    /** 1053: Audio Device Module: the underlying audio settings have changed. */
    BrtcWarningCodeAdmImproperSettings = 1053,
};

/** Error code.

Error codes occur when the SDK encounters an error that cannot be recovered automatically without any app intervention. For example, the SDK reports the BrtcErrorCodeStartCall = 1002 error if it fails to start a call, and reminds the user to call the [leaveChannel]([BrtcEngineKit leaveChannel:]) method.
*/
typedef NS_ENUM(NSInteger, BrtcErrorCode) {
    /** 0: No error occurs. */
    BrtcErrorCodeNoError = 0,
    /** 1: A general error occurs (no specified reason). */
    BrtcErrorCodeFailed = 1,
    /** 2: An invalid parameter is used. For example, the specific channel name includes illegal characters. */
    BrtcErrorCodeInvalidArgument = 2,
    /** 3: The SDK module is not ready.
     <p>Possible solutions：
     <ul><li>Check the audio device.</li>
     <li>Check the completeness of the app.</li>
     <li>Re-initialize the SDK.</li></ul></p>
    */
    BrtcErrorCodeNotReady = 3,
    /** 4: The current state of the SDK does not support this function. */
    BrtcErrorCodeNotSupported = 4,
    /** 5: The request is rejected. This is for internal SDK use only, and is not returned to the app through any method or callback. */
    BrtcErrorCodeRefused = 5,
    /** 6: The buffer size is not big enough to store the returned data. */
    BrtcErrorCodeBufferTooSmall = 6,
    /** 7: The SDK is not initialized before calling this method. */
    BrtcErrorCodeNotInitialized = 7,
    /** 9: No permission exists. This is for internal SDK use only, and is not returned to the app through any method or callback. */
    BrtcErrorCodeNoPermission = 9,
    /** 10: An API method timeout occurs. Some API methods require the SDK to return the execution result, and this error occurs if the request takes too long (over 10 seconds) for the SDK to process. */
    BrtcErrorCodeTimedOut = 10,
    /** 11: The request is canceled. This is for internal SDK use only, and is not returned to the app through any method or callback. */
    BrtcErrorCodeCanceled = 11,
    /** 12: The method is called too often. This is for internal SDK use only, and is not returned to the app through any method or callback. */
    BrtcErrorCodeTooOften = 12,
    /** 13: The SDK fails to bind to the network socket. This is for internal SDK use only, and is not returned to the app through any method or callback. */
    BrtcErrorCodeBindSocket = 13,
    /** 14: The network is unavailable. This is for internal SDK use only, and is not returned to the app through any method or callback. */
    BrtcErrorCodeNetDown = 14,
    /** 15: No network buffers are available. This is for internal SDK use only, and is not returned to the app through any method or callback. */
    BrtcErrorCodeNoBufs = 15,
    /** 17: The request to join the channel is rejected.
     <p>Possible reasons are:
     <ul><li>The user is already in the channel, and still calls the API method to join the channel, for example, [joinChannelByToken]([BrtcEngineKit joinChannelByToken:channelId:info:uid:joinSuccess:]).</li>
     <li>The user tries joining the channel during the echo test. Please join the channel after the echo test ends.</li></ul></p>
    */
    BrtcErrorCodeJoinChannelRejected = 17,
    /** 18: The request to leave the channel is rejected.
     <p>Possible reasons are:
     <ul><li>The user left the channel and still calls the API method to leave the channel, for example, [leaveChannel]([BrtcEngineKit leaveChannel:]).</li>
     <li>The user has not joined the channel and calls the API method to leave the channel.</li></ul></p>
    */
    BrtcErrorCodeLeaveChannelRejected = 18,
    /** 19: The resources are occupied and cannot be used. */
    BrtcErrorCodeAlreadyInUse = 19,
    /** 20: The SDK gave up the request due to too many requests.  */
    BrtcErrorCodeAbort = 20,
    /** 21: In Windows, specific firewall settings cause the SDK to fail to initialize and crash. */
    BrtcErrorCodeInitNetEngine = 21,
    /** 22: The app uses too much of the system resources and the SDK fails to allocate the resources. */
    BrtcErrorCodeResourceLimited = 22,
    /** 101: The specified App ID is invalid. Please try to rejoin the channel with a valid App ID.*/
    BrtcErrorCodeInvalidAppId = 101,
    /** 102: The specified channel name is invalid. Please try to rejoin the channel with a valid channel name. */
    BrtcErrorCodeInvalidChannelId = 102,
    /** 109: The token expired.
     <br></br><b>DEPRECATED</b> as of v2.4.1. Use BrtcConnectionChangedTokenExpired(9) in the `reason` parameter of [connectionChangedToState]([BrtcEngineDelegate rtcEngine:connectionChangedToState:reason:]).
     <p>Possible reasons are:
     <ul><li>Authorized Timestamp expired: The timestamp is represented by the number of seconds elapsed since 1/1/1970. The user can use the token to access the Brtc service within five minutes after the token is generated. If the user does not access the Brtc service after five minutes, this token is no longer valid.</li>
     <li>Call Expiration Timestamp expired: The timestamp is the exact time when a user can no longer use the Brtc service (for example, when a user is forced to leave an ongoing call). When a value is set for the Call Expiration Timestamp, it does not mean that the token will expire, but that the user will be banned from the channel.</li></ul></p>
     */
    BrtcErrorCodeTokenExpired = 109,
    /** 110: The token is invalid.
<br></br><b>DEPRECATED</b> as of v2.4.1. Use BrtcConnectionChangedInvalidToken(8) in the `reason` parameter of [connectionChangedToState]([BrtcEngineDelegate rtcEngine:connectionChangedToState:reason:]).
     <p>Possible reasons are:
     <ul><li>The App Certificate for the project is enabled in Dashboard, but the user is using the App ID. Once the App Certificate is enabled, the user must use a token.</li>
     <li>The uid is mandatory, and users must set the same uid as the one set in the [joinChannelByToken]([BrtcEngineKit joinChannelByToken:channelId:info:uid:joinSuccess:]) method.</li></ul></p>
     */
    BrtcErrorCodeInvalidToken = 110,
    /** 111: The Internet connection is interrupted. This applies to the Brtc Web SDK only. */
    BrtcErrorCodeConnectionInterrupted = 111,
    /** 112: The Internet connection is lost. This applies to the Brtc Web SDK only. */
    BrtcErrorCodeConnectionLost = 112,
    /** 113: The user is not in the channel when calling the [sendStreamMessage]([BrtcEngineKit sendStreamMessage:data:]) or [getUserInfoByUserAccount]([BrtcEngineKit getUserInfoByUserAccount:withError:]) method. */
    BrtcErrorCodeNotInChannel = 113,
    /** 114: The size of the sent data is over 1024 bytes when the user calls the [sendStreamMessage]([BrtcEngineKit sendStreamMessage:data:]) method. */
    BrtcErrorCodeSizeTooLarge = 114,
    /** 115: The bitrate of the sent data exceeds the limit of 6 Kbps when the user calls the [sendStreamMessage]([BrtcEngineKit sendStreamMessage:data:]) method. */
    BrtcErrorCodeBitrateLimit = 115,
    /** 116: Too many data streams (over five streams) are created when the user calls the [createDataStream]([BrtcEngineKit createDataStream:reliable:ordered:]) method. */
    BrtcErrorCodeTooManyDataStreams = 116,
    /** 120: Decryption fails. The user may have used a different encryption password to join the channel. Check your settings or try rejoining the channel. */
    BrtcErrorCodeDecryptionFailed = 120,
    /** 124: Incorrect watermark file parameter. */
    BrtcErrorCodeWatermarkParam = 124,
    /** 125: Incorrect watermark file path. */
    BrtcErrorCodeWatermarkPath = 125,
    /** 126: Incorrect watermark file format. */
    BrtcErrorCodeWatermarkPng = 126,
    /** 127: Incorrect watermark file information. */
    BrtcErrorCodeWatermarkInfo = 127,
    /** 128: Incorrect watermark file data format. */
    BrtcErrorCodeWatermarkAGRB = 128,
    /** 129: An error occurs in reading the watermark file. */
    BrtcErrorCodeWatermarkRead = 129,
    /** 130: The encrypted stream is not allowed to publish. */
    BrtcErrorCodeEncryptedStreamNotAllowedPublish = 130,
    /** 134: The user account is invalid. */
    BrtcErrorCodeInvalidUserAccount = 134,

    /** 151: CDN related errors. Remove the original URL address and add a new one by calling the [removePublishStreamUrl]([BrtcEngineKit removePublishStreamUrl:]) and [addPublishStreamUrl]([BrtcEngineKit addPublishStreamUrl:transcodingEnabled:]) methods. */
    BrtcErrorCodePublishStreamCDNError = 151,
    /** 152: The host publishes more than 10 URLs. Delete the unnecessary URLs before adding new ones. */
    BrtcErrorCodePublishStreamNumReachLimit = 152,
    /** 153: The host manipulates other hosts' URLs. Check your app logic. */
    BrtcErrorCodePublishStreamNotAuthorized = 153,
    /** 154: An error occurs in Brtc's streaming server. Call the [addPublishStreamUrl]([BrtcEngineKit addPublishStreamUrl:transcodingEnabled:]) method to publish the stream again. */
    BrtcErrorCodePublishStreamInternalServerError = 154,
    /** 155: The server fails to find the stream. */
    BrtcErrorCodePublishStreamNotFound = 155,
    /** 156: The format of the RTMP stream URL is not supported. Check whether the URL format is correct. */
    BrtcErrorCodePublishStreamFormatNotSuppported = 156,

    /** 1001: Fails to load the media engine. */
    BrtcErrorCodeLoadMediaEngine = 1001,
    /** 1002: Fails to start the call after enabling the media engine. */
    BrtcErrorCodeStartCall = 1002,
    /** 1003: Fails to start the camera.
     <br></br><b>DEPRECATED</b> as of v2.4.1. Use BrtcLocalVideoStreamErrorCaptureFailure(4) in the `error` parameter of [connectionChangedToState]([BrtcEngineDelegate rtcEngine:connectionChangedToState:reason:]).
     */
    BrtcErrorCodeStartCamera = 1003,
    /** 1004: Fails to start the video rendering module. */
    BrtcErrorCodeStartVideoRender = 1004,
    /** 1005: A general error occurs in the Audio Device Module (the reason is not classified specifically). Check if the audio device is used by another app, or try rejoining the channel. */
    BrtcErrorCodeAdmGeneralError = 1005,
    /** 1006: Audio Device Module: An error occurs in using the Java resources. */
    BrtcErrorCodeAdmJavaResource = 1006,
    /** 1007: Audio Device Module: An error occurs in setting the sampling frequency. */
    BrtcErrorCodeAdmSampleRate = 1007,
    /** 1008: Audio Device Module: An error occurs in initializing the playback device. */
    BrtcErrorCodeAdmInitPlayout = 1008,
    /** 1009: Audio Device Module: An error occurs in starting the playback device. */
    BrtcErrorCodeAdmStartPlayout = 1009,
    /** 1010: Audio Device Module: An error occurs in stopping the playback device. */
    BrtcErrorCodeAdmStopPlayout = 1010,
    /** 1011: Audio Device Module: An error occurs in initializing the recording device. */
    BrtcErrorCodeAdmInitRecording = 1011,
    /** 1012: Audio Device Module: An error occurs in starting the recording device. */
    BrtcErrorCodeAdmStartRecording = 1012,
    /** 1013: Audio Device Module: An error occurs in stopping the recording device. */
    BrtcErrorCodeAdmStopRecording = 1013,
    /** 1015: Audio Device Module: A playback error occurs. Check your playback device, or try rejoining the channel. */
    BrtcErrorCodeAdmRuntimePlayoutError = 1015,
    /** 1017: Audio Device Module: A recording error occurs. */
    BrtcErrorCodeAdmRuntimeRecordingError = 1017,
    /** 1018: Audio Device Module: Fails to record. */
    BrtcErrorCodeAdmRecordAudioFailed = 1018,
    /** 1020: Audio Device Module: Abnormal audio playback frequency. */
    BrtcErrorCodeAdmPlayAbnormalFrequency = 1020,
    /** 1021: Audio Device Module: Abnormal audio recording frequency. */
    BrtcErrorCodeAdmRecordAbnormalFrequency = 1021,
    /** 1022: Audio Device Module: An error occurs in initializing the loopback device. */
    BrtcErrorCodeAdmInitLoopback  = 1022,
    /** 1023: Audio Device Module: An error occurs in starting the loopback device. */
    BrtcErrorCodeAdmStartLoopback = 1023,
    /** 1027: Audio Device Module: An error occurs in no recording Permission. */
    BrtcErrorCodeAdmNoPermission = 1027,
    /** 1359: No recording device exists. */
    BrtcErrorCodeAdmNoRecordingDevice = 1359,
    /** 1360: No playback device exists. */
    BrtcErrorCodeAdmNoPlayoutDevice = 1360,
    /** 1501: Video Device Module: The camera is unauthorized. */
    BrtcErrorCodeVdmCameraNotAuthorized = 1501,
    /** 1600: Video Device Module: An unknown error occurs. */
    BrtcErrorCodeVcmUnknownError = 1600,
    /** 1601: Video Device Module: An error occurs in initializing the video encoder. */
    BrtcErrorCodeVcmEncoderInitError = 1601,
    /** 1602: Video Device Module: An error occurs in video encoding. */
    BrtcErrorCodeVcmEncoderEncodeError = 1602,
    /** 1603: Video Device Module: An error occurs in setting the video encoder.
    <p><b>DEPRECATED</b></p>
    */
    BrtcErrorCodeVcmEncoderSetError = 1603,
};

/** The state of the audio mixing file. */
typedef NS_ENUM(NSInteger, BrtcAudioMixingStateCode){
    /** The audio mixing file is playing. */
    BrtcAudioMixingStatePlaying = 710,
    /** The audio mixing file pauses playing. */
    BrtcAudioMixingStatePaused = 711,
    /** The audio mixing file stops playing. */
    BrtcAudioMixingStateStopped = 713,
    /** An exception occurs when playing the audio mixing file. */
    BrtcAudioMixingStateFailed = 714,
};

/**  The error code of the audio mixing file. */
typedef NS_ENUM(NSInteger, BrtcAudioMixingErrorCode){
    /** The SDK cannot open the audio mixing file. */
   BrtcAudioMixingErrorCanNotOpen = 701,
   /** The SDK opens the audio mixing file too frequently. */
   BrtcAudioMixingErrorTooFrequentCall = 702,
   /** The opening of the audio mixing file is interrupted. */
   BrtcAudioMixingErrorInterruptedEOF = 703,
   /** No error. */
   BrtcAudioMixingErrorOK = 0,
};

/** Video profile.

**DEPRECATED**

Please use BrtcVideoEncoderConfiguration.

iPhones do not support resolutions above 720p.
*/
typedef NS_ENUM(NSInteger, BrtcVideoProfile) {
    /** Invalid profile. */
    BrtcVideoProfileInvalid = -1,
    /** Resolution 160 &times; 120, frame rate 15 fps, bitrate 65 Kbps. */
    BrtcVideoProfileLandscape120P = 0,
    /** (iOS only) Resolution 120 &times; 120, frame rate 15 fps, bitrate 50 Kbps. */
    BrtcVideoProfileLandscape120P_3 = 2,
    /** (iOS only) Resolution 320 &times; 180, frame rate 15 fps, bitrate 140 Kbps. */
    BrtcVideoProfileLandscape180P = 10,
    /** (iOS only) Resolution 180 &times; 180, frame rate 15 fps, bitrate 100 Kbps. */
    BrtcVideoProfileLandscape180P_3 = 12,
    /** Resolution 240 &times; 180, frame rate 15 fps, bitrate 120 Kbps. */
    BrtcVideoProfileLandscape180P_4 = 13,
    /** Resolution 320 &times; 240, frame rate 15 fps, bitrate 200 Kbps. */
    BrtcVideoProfileLandscape240P = 20,
    /** (iOS only) Resolution 240 &times; 240, frame rate 15 fps, bitrate 140 Kbps. */
    BrtcVideoProfileLandscape240P_3 = 22,
    /** Resolution 424 &times; 240, frame rate 15 fps, bitrate 220 Kbps. */
    BrtcVideoProfileLandscape240P_4 = 23,
    /** Resolution 640 &times; 360, frame rate 15 fps, bitrate 400 Kbps. */
    BrtcVideoProfileLandscape360P = 30,
    /** (iOS only) Resolution 360 &times; 360, frame rate 15 fps, bitrate 260 Kbps. */
    BrtcVideoProfileLandscape360P_3 = 32,
    /** Resolution 640 &times; 360, frame rate 30 fps, bitrate 600 Kbps. */
    BrtcVideoProfileLandscape360P_4 = 33,
    /** Resolution 360 &times; 360, frame rate 30 fps, bitrate 400 Kbps. */
    BrtcVideoProfileLandscape360P_6 = 35,
    /** Resolution 480 &times; 360, frame rate 15 fps, bitrate 320 Kbps. */
    BrtcVideoProfileLandscape360P_7 = 36,
    /** Resolution 480 &times; 360, frame rate 30 fps, bitrate 490 Kbps. */
    BrtcVideoProfileLandscape360P_8 = 37,
    /** Resolution 640 &times; 360, frame rate 15 fps, bitrate 800 Kbps.
    <br>
     <b>Note:</b> This profile applies to the live broadcast channel profile only.
     */
    BrtcVideoProfileLandscape360P_9 = 38,
    /** Resolution 640 &times; 360, frame rate 24 fps, bitrate 800 Kbps.
    <br>
     <b>Note:</b> This profile applies to the live broadcast channel profile only.
     */
    BrtcVideoProfileLandscape360P_10 = 39,
    /** Resolution 640 &times; 360, frame rate 24 fps, bitrate 1000 Kbps.
    <br>
     <b>Note:</b> This profile applies to the live broadcast channel profile only.
     */
    BrtcVideoProfileLandscape360P_11 = 100,
    /** Resolution 640 &times; 480, frame rate 15 fps, bitrate 500 Kbps. */
    BrtcVideoProfileLandscape480P = 40,
    /** (iOS only) Resolution 480 &times; 480, frame rate 15 fps, bitrate 400 Kbps. */
    BrtcVideoProfileLandscape480P_3 = 42,
    /** Resolution 640 &times; 480, frame rate 30 fps, bitrate 750 Kbps. */
    BrtcVideoProfileLandscape480P_4 = 43,
    /** Resolution 480 &times; 480, frame rate 30 fps, bitrate 600 Kbps. */
    BrtcVideoProfileLandscape480P_6 = 45,
    /** Resolution 848 &times; 480, frame rate 15 fps, bitrate 610 Kbps. */
    BrtcVideoProfileLandscape480P_8 = 47,
    /** Resolution 848 &times; 480, frame rate 30 fps, bitrate 930 Kbps. */
    BrtcVideoProfileLandscape480P_9 = 48,
    /** Resolution 640 &times; 480, frame rate 10 fps, bitrate 400 Kbps. */
    BrtcVideoProfileLandscape480P_10 = 49,
    /** Resolution 1280 &times; 720, frame rate 15 fps, bitrate 1130 Kbps. */
    BrtcVideoProfileLandscape720P = 50,
    /** Resolution 1280 &times; 720, frame rate 30 fps, bitrate 1710 Kbps. */
    BrtcVideoProfileLandscape720P_3 = 52,
    /** Resolution 960 &times; 720, frame rate 15 fps, bitrate 910 Kbps. */
    BrtcVideoProfileLandscape720P_5 = 54,
    /** Resolution 960 &times; 720, frame rate 30 fps, bitrate 1380 Kbps. */
    BrtcVideoProfileLandscape720P_6 = 55,
    /** (macOS only) Resolution 1920 &times; 1080, frame rate 15 fps, bitrate 2080 Kbps. */
    BrtcVideoProfileLandscape1080P = 60,
    /** (macOS only) Resolution 1920 &times; 1080, frame rate 30 fps, bitrate 3150 Kbps. */
    BrtcVideoProfileLandscape1080P_3 = 62,
    /** (macOS only) Resolution 1920 &times; 1080, frame rate 60 fps, bitrate 4780 Kbps. */
    BrtcVideoProfileLandscape1080P_5 = 64,
    /** (macOS only) Resolution 2560 &times; 1440, frame rate 30 fps, bitrate 4850 Kbps. */
    BrtcVideoProfileLandscape1440P = 66,
    /** (macOS only) Resolution 2560 &times; 1440, frame rate 60 fps, bitrate 6500 Kbps. */
    BrtcVideoProfileLandscape1440P_2 = 67,
    /** (macOS only) Resolution 3840 &times; 2160, frame rate 30 fps, bitrate 6500 Kbps. */
    BrtcVideoProfileLandscape4K = 70,
    /** (macOS only) Resolution 3840 &times; 2160, frame rate 60 fps, bitrate 6500 Kbps. */
    BrtcVideoProfileLandscape4K_3 = 72,

    /** Resolution 120 &times; 160, frame rate 15 fps, bitrate 65 Kbps. */
    BrtcVideoProfilePortrait120P = 1000,
    /** (iOS only) Resolution 120 &times; 120, frame rate 15 fps, bitrate 50 Kbps. */
    BrtcVideoProfilePortrait120P_3 = 1002,
    /** (iOS only) Resolution 180 &times; 320, frame rate 15 fps, bitrate 140 Kbps. */
    BrtcVideoProfilePortrait180P = 1010,
    /** (iOS only) Resolution 180 &times; 180, frame rate 15 fps, bitrate 100 Kbps. */
    BrtcVideoProfilePortrait180P_3 = 1012,
    /** Resolution 180 &times; 240, frame rate 15 fps, bitrate 120 Kbps. */
    BrtcVideoProfilePortrait180P_4 = 1013,
    /** Resolution 240 &times; 320, frame rate 15 fps, bitrate 200 Kbps. */
    BrtcVideoProfilePortrait240P = 1020,
    /** (iOS only) Resolution 240 &times; 240, frame rate 15 fps, bitrate 140 Kbps. */
    BrtcVideoProfilePortrait240P_3 = 1022,
    /** Resolution 240 &times; 424, frame rate 15 fps, bitrate 220 Kbps. */
    BrtcVideoProfilePortrait240P_4 = 1023,
    /** Resolution 360 &times; 640, frame rate 15 fps, bitrate 400 Kbps. */
    BrtcVideoProfilePortrait360P = 1030,
    /** (iOS only) Resolution 360 &times; 360, frame rate 15 fps, bitrate 260 Kbps. */
    BrtcVideoProfilePortrait360P_3 = 1032,
    /** Resolution 360 &times; 640, frame rate 30 fps, bitrate 600 Kbps. */
    BrtcVideoProfilePortrait360P_4 = 1033,
    /** Resolution 360 &times; 360, frame rate 30 fps, bitrate 400 Kbps. */
    BrtcVideoProfilePortrait360P_6 = 1035,
    /** Resolution 360 &times; 480, frame rate 15 fps, bitrate 320 Kbps. */
    BrtcVideoProfilePortrait360P_7 = 1036,
    /** Resolution 360 &times; 480, frame rate 30 fps, bitrate 490 Kbps. */
    BrtcVideoProfilePortrait360P_8 = 1037,
    /** Resolution 360 &times; 640, frame rate 15 fps, bitrate 600 Kbps. */
    BrtcVideoProfilePortrait360P_9 = 1038,
    /** Resolution 360 &times; 640, frame rate 24 fps, bitrate 800 Kbps. */
    BrtcVideoProfilePortrait360P_10 = 1039,
    /** Resolution 360 &times; 640, frame rate 24 fps, bitrate 800 Kbps. */
    BrtcVideoProfilePortrait360P_11 = 1100,
    /** Resolution 480 &times; 640, frame rate 15 fps, bitrate 500 Kbps. */
    BrtcVideoProfilePortrait480P = 1040,
    /** (iOS only) Resolution 480 &times; 480, frame rate 15 fps, bitrate 400 Kbps. */
    BrtcVideoProfilePortrait480P_3 = 1042,
    /** Resolution 480 &times; 640, frame rate 30 fps, bitrate 750 Kbps. */
    BrtcVideoProfilePortrait480P_4 = 1043,
    /** Resolution 480 &times; 480, frame rate 30 fps, bitrate 600 Kbps. */
    BrtcVideoProfilePortrait480P_6 = 1045,
    /** Resolution 480 &times; 848, frame rate 15 fps, bitrate 610 Kbps. */
    BrtcVideoProfilePortrait480P_8 = 1047,
    /** Resolution 480 &times; 848, frame rate 30 fps, bitrate 930 Kbps. */
    BrtcVideoProfilePortrait480P_9 = 1048,
    /** Resolution 480 &times; 640, frame rate 10 fps, bitrate 400 Kbps. */
    BrtcVideoProfilePortrait480P_10 = 1049,
    /** Resolution 720 &times; 1280, frame rate 15 fps, bitrate 1130 Kbps. */
    BrtcVideoProfilePortrait720P = 1050,
    /** Resolution 720 &times; 1280, frame rate 30 fps, bitrate 1710 Kbps. */
    BrtcVideoProfilePortrait720P_3 = 1052,
    /** Resolution 720 &times; 960, frame rate 15 fps, bitrate 910 Kbps. */
    BrtcVideoProfilePortrait720P_5 = 1054,
    /** Resolution 720 &times; 960, frame rate 30 fps, bitrate 1380 Kbps. */
    BrtcVideoProfilePortrait720P_6 = 1055,
    /** (macOS only) Resolution 1080 &times; 1920, frame rate 15 fps, bitrate 2080 Kbps. */
    BrtcVideoProfilePortrait1080P = 1060,
    /** (macOS only) Resolution 1080 &times; 1920, frame rate 30 fps, bitrate 3150 Kbps. */
    BrtcVideoProfilePortrait1080P_3 = 1062,
    /** (macOS only) Resolution 1080 &times; 1920, frame rate 60 fps, bitrate 4780 Kbps. */
    BrtcVideoProfilePortrait1080P_5 = 1064,
    /** (macOS only) Resolution 1440 &times; 2560, frame rate 30 fps, bitrate 4850 Kbps. */
    BrtcVideoProfilePortrait1440P = 1066,
    /** (macOS only) Resolution 1440 &times; 2560, frame rate 60 fps, bitrate 6500 Kbps. */
    BrtcVideoProfilePortrait1440P_2 = 1067,
    /** (macOS only) Resolution 2160 &times; 3840, frame rate 30 fps, bitrate 6500 Kbps. */
    BrtcVideoProfilePortrait4K = 1070,
    /** (macOS only) Resolution 2160 &times; 3840, frame rate 60 fps, bitrate 6500 Kbps. */
    BrtcVideoProfilePortrait4K_3 = 1072,
    /** (Default) Resolution 640 &times; 360, frame rate 15 fps, bitrate 400 Kbps. */
    BrtcVideoProfileDEFAULT = BrtcVideoProfileLandscape360P,
};

/** The camera capturer configuration. */
typedef NS_ENUM(NSInteger, BrtcCameraCaptureOutputPreference) {
    /** (default) Self-adapts the camera output parameters to the system performance and network conditions to balance CPU consumption and video preview quality. */
    BrtcCameraCaptureOutputPreferenceAuto = 0,
    /** Prioritizes the system performance. The SDK chooses the dimension and frame rate of the local camera capture closest to those set by [setVideoEncoderConfiguration]([BrtcEngineKit setVideoEncoderConfiguration:]). */
    BrtcCameraCaptureOutputPreferencePerformance = 1,
    /** Prioritizes the local preview quality. The SDK chooses higher camera output parameters to improve the local video preview quality. This option requires extra CPU and RAM usage for video pre-processing. */
    BrtcCameraCaptureOutputPreferencePreview = 2,
    /** Internal use only */
    BrtcCameraCaptureOutputPreferenceUnkown = 3
};


/** Video frame rate */
typedef NS_ENUM(NSInteger, BrtcVideoFrameRate) {
    /** 1 fps. */
    BrtcVideoFrameRateFps1 = 1,
    /** 7 fps. */
    BrtcVideoFrameRateFps7 = 7,
    /** 10 fps. */
    BrtcVideoFrameRateFps10 = 10,
    /** 15 fps. */
    BrtcVideoFrameRateFps15 = 15,
    /** 24 fps. */
    BrtcVideoFrameRateFps24 = 24,
    /** 30 fps. */
    BrtcVideoFrameRateFps30 = 30,
    /** 60 fps (macOS only). */
    BrtcVideoFrameRateFps60 = 60,
};

/** Video output orientation mode.

  **Note:** When a custom video source is used, if you set BrtcVideoOutputOrientationMode as BrtcVideoOutputOrientationModeFixedLandscape(1) or BrtcVideoOutputOrientationModeFixedPortrait(2), when the rotated video image has a different orientation than the specified output orientation, the video encoder first crops it and then encodes it.
 */
typedef NS_ENUM(NSInteger, BrtcVideoOutputOrientationMode) {
    /** Adaptive mode (Default).
     <p>The video encoder adapts to the orientation mode of the video input device. When you use a custom video source, the output video from the encoder inherits the orientation of the original video.
     <ul><li>If the width of the captured video from the SDK is greater than the height, the encoder sends the video in landscape mode. The encoder also sends the rotational information of the video, and the receiver uses the rotational information to rotate the received video.</li>
     <li>If the original video is in portrait mode, the output video from the encoder is also in portrait mode. The encoder also sends the rotational information of the video to the receiver.</li></ul></p>
     */
    BrtcVideoOutputOrientationModeAdaptative = 0,
    /** Landscape mode.
     <p>The video encoder always sends the video in landscape mode. The video encoder rotates the original video before sending it and the rotational information is 0. This mode applies to scenarios involving CDN live streaming.</p>
     */
    BrtcVideoOutputOrientationModeFixedLandscape = 1,
     /** Portrait mode.
      <p>The video encoder always sends the video in portrait mode. The video encoder rotates the original video before sending it and the rotational information is 0. This mode applies to scenarios involving CDN live streaming.</p>
      */
    BrtcVideoOutputOrientationModeFixedPortrait = 2,
};

/** Channel profile. */
typedef NS_ENUM(NSInteger, BrtcChannelProfile) {
    /** Communication (default).
     <p>This is used in one-on-one or group calls, where all users in the channel can talk freely.</p>
     */
    BrtcChannelProfileCommunication = 0,
    /** Live Broadcast.
     <p>Host and audience roles that can be set by calling the [setClientRole]([BrtcEngineKit setClientRole:]) method. The host sends and receives voice/video, while the audience can only receive voice/video.</p>
     */
    BrtcChannelProfileLiveBroadcasting = 1,
    /** Gaming.
     <p>This mode is for the Brtc Gaming SDK only.</p>
     <p>Any user in the channel can talk freely. This mode uses the codec with low-power consumption and low bitrate by default. </p>
     */
    BrtcChannelProfileGame = 2,
};

/** Client role in a live broadcast. */
typedef NS_ENUM(NSInteger, BrtcClientRole) {
    /** Host. */
    BrtcClientRoleBroadcaster = 1,
    /** Audience. */
    BrtcClientRoleAudience = 2,
};


/** Media type. */
typedef NS_ENUM(NSInteger, BrtcMediaType) {
    /** No audio and video. */
    BrtcMediaTypeNone = 0,
    /** Audio only. */
    BrtcMediaTypeAudioOnly = 1,
    /** Video only. */
    BrtcMediaTypeVideoOnly = 2,
    /** Audio and video. */
    BrtcMediaTypeAudioAndVideo = 3,
};

/** Encryption mode */
typedef NS_ENUM(NSInteger, BrtcEncryptionMode) {
    /** When encryptionMode is set as NULL, the encryption mode is set as "aes-128-xts" by default. */
    BrtcEncryptionModeNone = 0,
    /** (Default) 128-bit AES encryption, XTS mode. */
    BrtcEncryptionModeAES128XTS = 1,
    /** 256-bit AES encryption, XTS mode. */
    BrtcEncryptionModeAES256XTS = 2,
    /** 128-bit AES encryption, ECB mode. */
    BrtcEncryptionModeAES128ECB = 3,
};

/** Reason for the user being offline. */
typedef NS_ENUM(NSUInteger, BrtcUserOfflineReason) {
    /** The user left the current channel. */
    BrtcUserOfflineReasonQuit = 0,
    /** The SDK timed out and the user dropped offline because no data packet is received within a certain period of time. If a user quits the call and the message is not passed to the SDK (due to an unreliable channel), the SDK assumes the user dropped offline. */
    BrtcUserOfflineReasonDropped = 1,
    /** (Live broadcast only.) The client role switched from the host to the audience. */
    BrtcUserOfflineReasonBecomeAudience = 2,
};

/** The RTMP streaming state. */
typedef NS_ENUM(NSUInteger, BrtcRtmpStreamingState) {
  /** The RTMP streaming has not started or has ended. This state is also triggered after you remove an RTMP address from the CDN by calling removePublishStreamUrl.*/
  BrtcRtmpStreamingStateIdle = 0,
  /** The SDK is connecting to Brtc's streaming server and the RTMP server. This state is triggered after you call the [addPublishStreamUrl]([BrtcEngineKit addPublishStreamUrl:transcodingEnabled:]) method. */
  BrtcRtmpStreamingStateConnecting = 1,
  /** The RTMP streaming is being published. The SDK successfully publishes the RTMP streaming and returns this state. */
  BrtcRtmpStreamingStateRunning = 2,
  /** The RTMP streaming is recovering. When exceptions occur to the CDN, or the streaming is interrupted, the SDK attempts to resume RTMP streaming and returns this state.
<li> If the SDK successfully resumes the streaming, BrtcRtmpStreamingStateRunning(2) returns.
<li> If the streaming does not resume within 60 seconds or server errors occur, BrtcRtmpStreamingStateFailure(4) returns. You can also reconnect to the server by calling the [removePublishStreamUrl]([BrtcEngineKit removePublishStreamUrl:]) and [addPublishStreamUrl]([BrtcEngineKit addPublishStreamUrl:transcodingEnabled:]) methods. */
  BrtcRtmpStreamingStateRecovering = 3,
  /** The RTMP streaming fails. See the errorCode parameter for the detailed error information. You can also call the [addPublishStreamUrl]([BrtcEngineKit addPublishStreamUrl:transcodingEnabled:]) method to publish the RTMP streaming again. */
  BrtcRtmpStreamingStateFailure = 4,
};


enum RTMP_STREAM_PUBLISH_ERROR
{
  RTMP_STREAM_PUBLISH_ERROR_OK = 0,
  RTMP_STREAM_PUBLISH_ERROR_INVALID_ARGUMENT = 1,
  RTMP_STREAM_PUBLISH_ERROR_ENCRYPTED_STREAM_NOT_ALLOWED = 2,
  RTMP_STREAM_PUBLISH_ERROR_CONNECTION_TIMEOUT = 3,
  RTMP_STREAM_PUBLISH_ERROR_INTERNAL_SERVER_ERROR = 4,
  RTMP_STREAM_PUBLISH_ERROR_RTMP_SERVER_ERROR = 5,
  RTMP_STREAM_PUBLISH_ERROR_TOO_OFTEN = 6,
  RTMP_STREAM_PUBLISH_ERROR_REACH_LIMIT = 7,
  RTMP_STREAM_PUBLISH_ERROR_NOT_AUTHORIZED = 8,
  RTMP_STREAM_PUBLISH_ERROR_STREAM_NOT_FOUND = 9,
  RTMP_STREAM_PUBLISH_ERROR_FORMAT_NOT_SUPPORTED = 10,
};

/** The detailed error information for streaming. */
typedef NS_ENUM(NSUInteger, BrtcRtmpStreamingErrorCode) {
  /** The RTMP streaming publishes successfully. */
  BrtcRtmpStreamingErrorCodeOK = 0,
  /** Invalid argument used. If, for example, you do not call the [setLiveTranscoding]([BrtcEngineKit setLiveTranscoding:]) method to configure the LiveTranscoding parameters before calling the [addPublishStreamUrl]([BrtcEngineKit addPublishStreamUrl:transcodingEnabled:]) method, the SDK returns this error. Check whether you set the parameters in the setLiveTranscoding method properly. */
  BrtcRtmpStreamingErrorCodeInvalidParameters = 1,
  /** The RTMP streaming is encrypted and cannot be published. */
  BrtcRtmpStreamingErrorCodeEncryptedStreamNotAllowed = 2,
  /** Timeout for the RTMP streaming. Call the [addPublishStreamUrl]([BrtcEngineKit addPublishStreamUrl:transcodingEnabled:]) method to publish the streaming again. */
  BrtcRtmpStreamingErrorCodeConnectionTimeout = 3,
  /** An error occurs in Brtc's streaming server. Call the [addPublishStreamUrl]([BrtcEngineKit addPublishStreamUrl:transcodingEnabled:]) method to publish the streaming again. */
  BrtcRtmpStreamingErrorCodeInternalServerError = 4,
  /** An error occurs in the RTMP server. */
  BrtcRtmpStreamingErrorCodeRtmpServerError = 5,
  /** The RTMP streaming publishes too frequently. */
  BrtcRtmpStreamingErrorCodeTooOften = 6,
  /** The host publishes more than 10 URLs. Delete the unnecessary URLs before adding new ones. */
  BrtcRtmpStreamingErrorCodeReachLimit = 7,
  /** The host manipulates other hosts' URLs. Check your app logic. */
  BrtcRtmpStreamingErrorCodeNotAuthorized = 8,
  /** Brtc's server fails to find the RTMP streaming. */
  BrtcRtmpStreamingErrorCodeStreamNotFound = 9,
  /** The format of the RTMP streaming URL is not supported. Check whether the URL format is correct. */
  BrtcRtmpStreamingErrorCodeFormatNotSupported = 10,
};

/** State of importing an external video stream in a live broadcast. */
typedef NS_ENUM(NSUInteger, BrtcInjectStreamStatus) {
    /** The external video stream imported successfully. */
    BrtcInjectStreamStatusStartSuccess = 0,
    /** The external video stream already exists. */
    BrtcInjectStreamStatusStartAlreadyExists = 1,
    /** The external video stream import is unauthorized. */
    BrtcInjectStreamStatusStartUnauthorized = 2,
    /** Import external video stream timeout. */
    BrtcInjectStreamStatusStartTimedout = 3,
    /** The external video stream failed to import. */
    BrtcInjectStreamStatusStartFailed = 4,
    /** The external video stream imports successfully. */
    BrtcInjectStreamStatusStopSuccess = 5,
    /** No external video stream is found. */
    BrtcInjectStreamStatusStopNotFound = 6,
    /** The external video stream is stopped from being unauthorized. */
    BrtcInjectStreamStatusStopUnauthorized = 7,
    /** Importing the external video stream timeout. */
    BrtcInjectStreamStatusStopTimedout = 8,
    /** Importing the external video stream failed. */
    BrtcInjectStreamStatusStopFailed = 9,
    /** The external video stream import is interrupted. */
    BrtcInjectStreamStatusBroken = 10,
};

/** Output log filter level. */
typedef NS_ENUM(NSUInteger, BrtcLogFilter) {
    /** Do not output any log information. */
    BrtcLogFilterOff = 0,
    /** Output all log information. Set your log filter as debug if you want to get the most complete log file. */
    BrtcLogFilterDebug = 0x080f,
    /** Output CRITICAL, ERROR, WARNING, and INFO level log information. We recommend setting your log filter as this level. */
    BrtcLogFilterInfo = 0x000f,
    /** Outputs CRITICAL, ERROR, and WARNING level log information. */
    BrtcLogFilterWarning = 0x000e,
    /** Outputs CRITICAL and ERROR level log information. */
    BrtcLogFilterError = 0x000c,
    /** Outputs CRITICAL level log information. */
    BrtcLogFilterCritical = 0x0008,
};

/** Audio recording quality. */
typedef NS_ENUM(NSInteger, BrtcAudioRecordingQuality) {
   /** Low quality: The sample rate is 32 KHz, and the file size is around 1.2 MB after 10 minutes of recording. */
    BrtcAudioRecordingQualityLow = 0,
    /** Medium quality: The sample rate is 32 KHz, and the file size is around 2 MB after 10 minutes of recording. */
    BrtcAudioRecordingQualityMedium = 1,
    /** High quality: The sample rate is 32 KHz, and the file size is around 3.75 MB after 10 minutes of recording. */
    BrtcAudioRecordingQualityHigh = 2
};

/** Audio recording position. */
typedef NS_ENUM(NSInteger, BrtcAudioRecordingPosition) {
   /** Mixed recoding and playback: The SDK will record the voices of all users in the channel. */
    BrtcAudioRecordingPositionMixedRecordingAndPlayback = 0,
    /** Only recording: The SDK will record the voice of the local user. */
    BrtcAudioRecordingPositionRecording = 1,
    /** Only mixed playback: The SDK will record the voices of remote users. */
    BrtcAudioRecordingPositionMixedPlayback = 2
};


/** Lifecycle of the CDN live video stream.

**DEPRECATED**
*/
typedef NS_ENUM(NSInteger, BrtcRtmpStreamLifeCycle) {
    /** Bound to the channel lifecycle. If all hosts leave the channel, the CDN live streaming stops after 30 seconds. */
    BrtcRtmpStreamLifeCycleBindToChannel = 1,
    /** Bound to the owner of the RTMP stream. If the owner leaves the channel, the CDN live streaming stops immediately. */
    BrtcRtmpStreamLifeCycleBindToOwnner = 2,
};

/** Network quality. */
typedef NS_ENUM(NSUInteger, BrtcNetworkQuality) {
    /** The network quality is unknown. */
    BrtcNetworkQualityUnknown = 0,
    /**  The network quality is excellent. */
    BrtcNetworkQualityExcellent = 1,
    /** The network quality is quite good, but the bitrate may be slightly lower than excellent. */
    BrtcNetworkQualityGood = 2,
    /** Users can feel the communication slightly impaired. */
    BrtcNetworkQualityPoor = 3,
    /** Users can communicate only not very smoothly. */
    BrtcNetworkQualityBad = 4,
     /** The network quality is so bad that users can hardly communicate. */
    BrtcNetworkQualityVBad = 5,
     /** The network is disconnected and users cannot communicate at all. */
    BrtcNetworkQualityDown = 6,
     /** Users cannot detect the network quality. (Not in use.) */
    BrtcNetworkQualityUnsupported = 7,
     /** Detecting the network quality. */
    BrtcNetworkQualityDetecting = 8,
};

/** Video stream type. */
typedef NS_ENUM(NSInteger, BrtcVideoStreamType) {
    /** High-bitrate, high-resolution video stream. */
    BrtcVideoStreamTypeHigh = 0,
    /** Low-bitrate, low-resolution video stream. */
    BrtcVideoStreamTypeLow = 1,
};

/** The priority of the remote user. */
typedef NS_ENUM(NSInteger, BrtcUserPriority) {
  /** The user's priority is high. */
  BrtcUserPriorityHigh = 50,
  /** (Default) The user's priority is normal. */
  BrtcUserPriorityNormal = 100,
};

/**  Quality change of the local video in terms of target frame rate and target bit rate since last count. */
typedef NS_ENUM(NSInteger, BrtcVideoQualityAdaptIndication) {
  /** The quality of the local video stays the same. */
  BrtcVideoQualityAdaptNone = 0,
  /** The quality improves because the network bandwidth increases. */
  BrtcVideoQualityAdaptUpBandwidth = 1,
  /** The quality worsens because the network bandwidth decreases. */
  BrtcVideoQualityAdaptDownBandwidth = 2,
};

/** Video display mode. */
typedef NS_ENUM(NSUInteger, BrtcVideoRenderMode) {
    /** Hidden(1): Uniformly scale the video until it fills the visible boundaries (cropped). One dimension of the video may have clipped contents. */
    BrtcVideoRenderModeHidden = 1,

    /** Fit(2): Uniformly scale the video until one of its dimension fits the boundary (zoomed to fit). Areas that are not filled due to the disparity in the aspect ratio are filled with black. */
    BrtcVideoRenderModeFit = 2,

    /**
     Adaptive(3)：This mode is deprecated.
     */
    BrtcVideoRenderModeAdaptive __deprecated_enum_msg("BrtcVideoRenderModeAdaptive is deprecated.") = 3,
};

/** Self-defined video codec profile. */
typedef NS_ENUM(NSInteger, BrtcVideoCodecProfileType) {
    /** 66: Baseline video codec profile. Generally used in video calls on mobile phones. */
    BrtcVideoCodecProfileTypeBaseLine = 66,
    /** 77: Main video codec profile. Generally used in mainstream electronics, such as MP4 players, portable video players, PSP, and iPads. */
    BrtcVideoCodecProfileTypeMain = 77,
    /** 100: (Default) High video codec profile. Generally used in high-resolution broadcasts or television. */
    BrtcVideoCodecProfileTypeHigh = 100
};

/** Video codec types. */
typedef NS_ENUM(NSInteger, BrtcVideoCodecType) {
    /** Standard VP8. */
    BrtcVideoCodecTypeVP8 = 1,
    /** Standard H264. */
    BrtcVideoCodecTypeH264 = 2,
    /** Enhanced VP8. */
    BrtcVideoCodecTypeEVP = 3,
    /** Enhanced H264. */
    BrtcVideoCodecTypeE264 = 4,
};

/** Self-defined video codec type */
typedef NS_ENUM(NSInteger, BrtcVideoCodecTranscodingType) {
    /** 1: (Default value)H264 */
    BrtcVideoCodecTranscodingTypeH264 = 1,
    /** 2: H265 */
    BrtcVideoCodecTranscodingTypeH265 = 2,
};

/** Video mirror mode. */
typedef NS_ENUM(NSUInteger, BrtcVideoMirrorMode) {
    /** The SDK determines the default mirror mode. */
    BrtcVideoMirrorModeAuto = 0,
    /** Enables mirror mode. */
    BrtcVideoMirrorModeEnabled = 1,
    /** Disables mirror mode. */
    BrtcVideoMirrorModeDisabled = 2,
};

/** Stream publish mode. */
typedef NS_ENUM(NSUInteger, BrtcStreamPublishState) {
    BrtcStreamPublishIdle = 0,
    BrtcStreamPublishNoPublished = 1,
    BrtcStreamPublishPublishing = 2,
    BrtcStreamPublishPublished = 3,
};

/** Stream subscribe mode. */
typedef NS_ENUM(NSUInteger, BrtcStreamSubscribeState) {
    BrtcStreamSubscribeIdle = 0,
    BrtcStreamSubscribeNoSubscribed = 1,
    BrtcStreamSubscribeSubscribing = 2,
    BrtcStreamSubscribeSubscribed = 3,
};

/** The content hint for screen sharing. */
typedef NS_ENUM(NSUInteger, BrtcVideoContentHint) {
    /** (Default) No content hint. */
    BrtcVideoContentHintNone = 0,
    /** Motion-intensive content. Choose this option if you prefer smoothness or when you are sharing a video clip, movie, or video game. */
    BrtcVideoContentHintMotion = 1,
    /** Motionless content. Choose this option if you prefer sharpness or when you are sharing a picture, PowerPoint slide, or text. */
    BrtcVideoContentHintDetails = 2,
};

/** The state of the remote video. */
typedef NS_ENUM(NSUInteger, BrtcVideoRemoteState) {
    /** 0: The remote video is in the default state, probably due to BrtcVideoRemoteStateReasonLocalMuted(3), BrtcVideoRemoteStateReasonRemoteMuted(5), or BrtcVideoRemoteStateReasonRemoteOffline(7).
     */
    BrtcVideoRemoteStateStopped = 0,
    /** 1: The first remote video packet is received.
     */
    BrtcVideoRemoteStateStarting = 1,
    /** 2: The remote video stream is decoded and plays normally, probably due to BrtcVideoRemoteStateReasonNetworkRecovery(2), BrtcVideoRemoteStateReasonLocalUnmuted(4), BrtcVideoRemoteStateReasonRemoteUnmuted(6), or BrtcVideoRemoteStateReasonAudioFallbackRecovery(9).
     */
    BrtcVideoRemoteStateDecoding = 2,
    /** 3: The remote video is frozen, probably due to BrtcVideoRemoteStateReasonNetworkCongestion(1) or BrtcVideoRemoteStateReasonAudioFallback(8).
     */
    BrtcVideoRemoteStateFrozen = 3,
    /** 4: The remote video fails to start, probably due to BrtcVideoRemoteStateReasonInternal(0).
     */
    BrtcVideoRemoteStateFailed = 4,
};

/** The reason of the remote video state change. */
typedef NS_ENUM(NSUInteger, BrtcVideoRemoteStateReason) {
    /** 0: Internal reasons. */
    BrtcVideoRemoteStateReasonInternal = 0,
    /** 1: Network congestion. */
    BrtcVideoRemoteStateReasonNetworkCongestion = 1,
    /** 2: Network recovery. */
    BrtcVideoRemoteStateReasonNetworkRecovery = 2,
    /** 3: The local user stops receiving the remote video stream or disables the video module. */
    BrtcVideoRemoteStateReasonLocalMuted = 3,
    /** 4: The local user resumes receiving the remote video stream or enables the video module. */
    BrtcVideoRemoteStateReasonLocalUnmuted = 4,
    /** 5: The remote user stops sending the video stream or disables the video module. */
    BrtcVideoRemoteStateReasonRemoteMuted = 5,
    /** 6: The remote user resumes sending the video stream or enables the video module. */
    BrtcVideoRemoteStateReasonRemoteUnmuted = 6,
    /** 7: The remote user leaves the channel. */
    BrtcVideoRemoteStateReasonRemoteOffline = 7,
    /** 8: The remote media stream falls back to the audio-only stream due to poor network conditions. */
    BrtcVideoRemoteStateReasonAudioFallback = 8,
    /** 9: The remote media stream switches back to the video stream after the network conditions improve. */
    BrtcVideoRemoteStateReasonAudioFallbackRecovery = 9,
};

/** Stream fallback option. */
typedef NS_ENUM(NSInteger, BrtcStreamFallbackOptions) {
    /** No fallback behavior for the local/remote video stream when the uplink/downlink network condition is unreliable. The quality of the stream is not guaranteed. */
    BrtcStreamFallbackOptionDisabled = 0,
    /** Under unreliable downlink network conditions, the remote video stream falls back to the low-stream (low resolution and low bitrate) video. You can only set this option in the [setRemoteSubscribeFallbackOption]([BrtcEngineKit setRemoteSubscribeFallbackOption:]) method. Nothing happens when you set this in the [setLocalPublishFallbackOption]([BrtcEngineKit setLocalPublishFallbackOption:]) method. */
    BrtcStreamFallbackOptionVideoStreamLow = 1,
    /**
     <li> Under unreliable uplink network conditions, the published video stream falls back to audio only.
     <li> Under unreliable downlink network conditions, the remote video stream first falls back to the low-stream (low resolution and low bitrate) video; and then to an audio-only stream if the network condition deteriorates. */
    BrtcStreamFallbackOptionAudioOnly = 2,
};

/** Audio sample rate. */
typedef NS_ENUM(NSInteger, BrtcAudioSampleRateType) {
    /** 32 kHz. */
    BrtcAudioSampleRateType32000 = 32000,
    /** 44.1 kHz. */
    BrtcAudioSampleRateType44100 = 44100,
    /** 48 kHz. */
    BrtcAudioSampleRateType48000 = 48000,
};

/** Audio profile. */
typedef NS_ENUM(NSInteger, BrtcAudioProfile) {
    /** Default audio profile. In the communication profile, the default value is BrtcAudioProfileSpeechStandard; in the live-broadcast profile, the default value is BrtcAudioProfileMusicStandard. */
    BrtcAudioProfileDefault = 0,
    /** A sample rate of 32 kHz, audio encoding, mono, and a bitrate of up to 18 Kbps. */
    BrtcAudioProfileSpeechStandard = 1,
    /** A sample rate of 48 kHz, music encoding, mono, and a bitrate of up to 48 Kbps. */
    BrtcAudioProfileMusicStandard = 2,
    /** A sample rate of 48 kHz, music encoding, stereo, and a bitrate of up to 56 Kbps. */
    BrtcAudioProfileMusicStandardStereo = 3,
    /** A sample rate of 48 kHz, music encoding, mono, and a bitrate of up to 128 Kbps. */
    BrtcAudioProfileMusicHighQuality = 4,
    /** A sample rate of 48 kHz, music encoding, stereo, and a bitrate of up to 192 Kbps. */
    BrtcAudioProfileMusicHighQualityStereo = 5,
};

/** Audio scenario. */
typedef NS_ENUM(NSInteger, BrtcAudioScenario) {
    /** Default. */
    BrtcAudioScenarioDefault = 0,
    /** Entertainment scenario, supporting voice during gameplay. */
    BrtcAudioScenarioChatRoomEntertainment = 1,
    /** Education scenario, prioritizing fluency and stability. */
    BrtcAudioScenarioEducation = 2,
    /** Live gaming scenario, enabling the gaming audio effects in the speaker mode in a live broadcast scenario. Choose this scenario for high-fidelity music playback.*/
    BrtcAudioScenarioGameStreaming = 3,
    /** Showroom scenario, optimizing the audio quality with external professional equipment. */
    BrtcAudioScenarioShowRoom = 4,
    /** Gaming scenario. */
    BrtcAudioScenarioChatRoomGaming = 5
};

/** Audio output routing. */
typedef NS_ENUM(NSInteger, BrtcAudioOutputRouting) {
    /** Default. */
    BrtcAudioOutputRoutingDefault = -1,
    /** Headset.*/
    BrtcAudioOutputRoutingHeadset = 0,
    /** Earpiece. */
    BrtcAudioOutputRoutingEarpiece = 1,
    /** Headset with no microphone. */
    BrtcAudioOutputRoutingHeadsetNoMic = 2,
    /** Speakerphone. */
    BrtcAudioOutputRoutingSpeakerphone = 3,
    /** Loudspeaker. */
    BrtcAudioOutputRoutingLoudspeaker = 4,
    /** Bluetooth headset. */
    BrtcAudioOutputRoutingHeadsetBluetooth = 5
};

/** Use mode of the onRecordAudioFrame callback. */
typedef NS_ENUM(NSInteger, BrtcAudioRawFrameOperationMode) {
    /** Read-only mode: Users only read the AudioFrame data without modifying anything. For example, when users acquire data with the Brtc SDK then push the RTMP streams. */
    BrtcAudioRawFrameOperationModeReadOnly = 0,
    /** Write-only mode: Users replace the AudioFrame data with their own data and pass them to the SDK for encoding. For example, when users acquire data. */
    BrtcAudioRawFrameOperationModeWriteOnly = 1,
    /** Read and write mode: Users read the data from AudioFrame, modify it, and then play it. For example, when users have their own sound-effect processing module and perform some voice pre-processing such as a voice change. */
    BrtcAudioRawFrameOperationModeReadWrite = 2,
};

/** Audio equalization band frequency. */
typedef NS_ENUM(NSInteger, BrtcAudioEqualizationBandFrequency) {
    /** 31 Hz. */
    BrtcAudioEqualizationBand31 = 0,
    /** 62 Hz. */
    BrtcAudioEqualizationBand62 = 1,
    /** 125 Hz. */
    BrtcAudioEqualizationBand125 = 2,
    /** 250 Hz. */
    BrtcAudioEqualizationBand250 = 3,
    /** 500 Hz */
    BrtcAudioEqualizationBand500 = 4,
    /** 1 kHz. */
    BrtcAudioEqualizationBand1K = 5,
    /** 2 kHz. */
    BrtcAudioEqualizationBand2K = 6,
    /** 4 kHz. */
    BrtcAudioEqualizationBand4K = 7,
    /** 8 kHz. */
    BrtcAudioEqualizationBand8K = 8,
    /** 16 kHz. */
    BrtcAudioEqualizationBand16K = 9,
};

/** Audio reverberation type. */
typedef NS_ENUM(NSInteger, BrtcAudioReverbType) {
    /** The level of the dry signal (dB). The value ranges between -20 and 10. */
    BrtcAudioReverbDryLevel = 0,
    /** The level of the early reflection signal (wet signal) in dB. The value ranges between -20 and 10. */
    BrtcAudioReverbWetLevel = 1,
    /** The room size of the reverberation. A larger room size means a stronger reverberation. The value ranges between 0 and 100. */
    BrtcAudioReverbRoomSize = 2,
    /** The length of the initial delay of the wet signal (ms). The value ranges between 0 and 200. */
    BrtcAudioReverbWetDelay = 3,
     /** The reverberation strength. The value ranges between 0 and 100. */
    BrtcAudioReverbStrength = 4,
};

/** The preset audio voice configuration used to change the voice effect. */
typedef NS_ENUM(NSInteger, BrtcAudioVoiceChanger) {
    /** The original voice (no local voice change). */
    BrtcAudioVoiceChangerOff = 0,
    /** An old man's voice. */
    BrtcAudioVoiceChangerOldMan = 1,
    /** A little boy's voice. */
    BrtcAudioVoiceChangerBabyBoy = 2,
    /** A little girl's voice. */
    BrtcAudioVoiceChangerBabyGirl = 3,
    /** TBD */
    BrtcAudioVoiceChangerZhuBaJie = 4,
    /** Ethereal vocal effects. */
    BrtcAudioVoiceChangerEthereal = 5,
    /** Hulk's voice. */
    BrtcAudioVoiceChangerHulk = 6
};

/** The preset local voice reverberation option. */
typedef NS_ENUM(NSInteger, BrtcAudioReverbPreset) {
    /** The original voice (no local voice reverberation). */
    BrtcAudioReverbPresetOff = 0,
    /** Pop music */
    BrtcAudioReverbPresetPopular = 1,
    /** R&B */
    BrtcAudioReverbPresetRnB = 2,
    /** Rock music */
    BrtcAudioReverbPresetRock = 3,
    /** Hip-hop music */
    BrtcAudioReverbPresetHipHop = 4,
    /** Pop concert */
    BrtcAudioReverbPresetVocalConcert = 5,
    /** Karaoke */
    BrtcAudioReverbPresetKTV = 6,
    /** Recording studio */
    BrtcAudioReverbPresetStudio = 7
};

/** Audio session restriction. */
typedef NS_OPTIONS(NSUInteger, BrtcAudioSessionOperationRestriction) {
    /** No restriction, the SDK has full control of the audio session operations. */
    BrtcAudioSessionOperationRestrictionNone              = 0,
    /** The SDK does not change the audio session category. */
    BrtcAudioSessionOperationRestrictionSetCategory       = 1,
    /** The SDK does not change any setting of the audio session (category, mode, categoryOptions). */
    BrtcAudioSessionOperationRestrictionConfigureSession  = 1 << 1,
    /** The SDK keeps the audio session active when leaving a channel. */
    BrtcAudioSessionOperationRestrictionDeactivateSession = 1 << 2,
    /** The SDK does not configure the audio session anymore. */
    BrtcAudioSessionOperationRestrictionAll               = 1 << 7
};

/** Audio codec profile. */
typedef NS_ENUM(NSInteger, BrtcAudioCodecProfileType) {
    /** (Default) LC-AAC, the low-complexity audio codec profile. */
  BrtcAudioCodecProfileLCAAC = 0,
  /** HE-AAC, the high-efficiency audio codec profile. */
  BrtcAudioCodecProfileHEAAC = 1
};

/** The state of the remote audio. */
typedef NS_ENUM(NSUInteger, BrtcAudioRemoteState) {
    /** 0: The remote audio is in the default state, probably due to BrtcAudioRemoteReasonLocalMuted(3), BrtcAudioRemoteReasonRemoteMuted(5), or BrtcAudioRemoteReasonRemoteOffline(7). */
    BrtcAudioRemoteStateStopped = 0,
    /** 1: The first remote audio packet is received. */
    BrtcAudioRemoteStateStarting = 1,
    /** 2: The remote audio stream is decoded and plays normally, probably due to BrtcAudioRemoteReasonNetworkRecovery(2), BrtcAudioRemoteReasonLocalUnmuted(4), or BrtcAudioRemoteReasonRemoteUnmuted(6). */
    BrtcAudioRemoteStateDecoding = 2,
    /** 3: The remote audio is frozen, probably due to BrtcAudioRemoteReasonNetworkCongestion(1). */
    BrtcAudioRemoteStateFrozen = 3,
    /** 4: The remote audio fails to start, probably due to BrtcAudioRemoteReasonInternal(0). */
    BrtcAudioRemoteStateFailed = 4,
};

/** The reason of the remote audio state change. */
typedef NS_ENUM(NSUInteger, BrtcAudioRemoteStateReason) {
    /** 0: Internal reasons. */
    BrtcAudioRemoteReasonInternal = 0,
    /** 1: Network congestion. */
    BrtcAudioRemoteReasonNetworkCongestion = 1,
    /** 2: Network recovery. */
    BrtcAudioRemoteReasonNetworkRecovery = 2,
    /** 3: The local user stops receiving the remote audio stream or disables the audio module. */
    BrtcAudioRemoteReasonLocalMuted = 3,
    /** 4: The local user resumes receiving the remote audio stream or enables the audio module. */
    BrtcAudioRemoteReasonLocalUnmuted = 4,
    /** 5: The remote user stops sending the audio stream or disables the audio module. */
    BrtcAudioRemoteReasonRemoteMuted = 5,
    /** 6: The remote user resumes sending the audio stream or enables the audio module. */
    BrtcAudioRemoteReasonRemoteUnmuted = 6,
    /** 7: The remote user leaves the channel. */
    BrtcAudioRemoteReasonRemoteOffline = 7,
};

/** The state of the local audio. */
typedef NS_ENUM(NSUInteger, BrtcAudioLocalState) {
    /** 0: The local audio is in the initial state. */
    BrtcAudioLocalStateStopped = 0,
    /** 1: The recording device starts successfully.  */
    BrtcAudioLocalStateRecording = 1,
    /** 2: The first audio frame encodes successfully. */
    BrtcAudioLocalStateEncoding = 2,
    /** 3: The local audio fails to start. */
    BrtcAudioLocalStateFailed = 3,
};

/** The error information of the local audio. */
typedef NS_ENUM(NSUInteger, BrtcAudioLocalError) {
    /** 0: The local audio is normal. */
    BrtcAudioLocalErrorOk = 0,
    /** 1: No specified reason for the local audio failure. */
    BrtcAudioLocalErrorFailure = 1,
    /** 2: No permission to use the local audio device. */
    BrtcAudioLocalErrorDeviceNoPermission = 2,
    /** 3: The microphone is in use. */
    BrtcAudioLocalErrorDeviceBusy = 3,
    /** 4: The local audio recording fails. Check whether the recording device is working properly. */
    BrtcAudioLocalErrorRecordFailure = 4,
    /** 5: The local audio encoding fails. */
    BrtcAudioLocalErrorEncodeFailure = 5,
};

/** Media device type. */
typedef NS_ENUM(NSInteger, BrtcMediaDeviceType) {
    /** Unknown device. */
    BrtcMediaDeviceTypeAudioUnknown = -1,
    /** Audio playback device. */
    BrtcMediaDeviceTypeAudioPlayout = 0,
    /** Audio recording device. */
    BrtcMediaDeviceTypeAudioRecording = 1,
    /** Video render device. */
    BrtcMediaDeviceTypeVideoRender = 2,
    /** Video capture device. */
    BrtcMediaDeviceTypeVideoCapture = 3,
};

/** Connection states. */
typedef NS_ENUM(NSInteger, BrtcConnectionStateType) {
    /** <p>1: The SDK is disconnected from Brtc's edge server.</p>
This is the initial state before [joinChannelByToken]([BrtcEngineKit joinChannelByToken:channelId:info:uid:joinSuccess:]).<br>
The SDK also enters this state when the app calls [leaveChannel]([BrtcEngineKit leaveChannel:]).
    */
    BrtcConnectionStateDisconnected = 1,
    /** <p>2: The SDK is connecting to Brtc's edge server.</p>
When the app calls [joinChannelByToken]([BrtcEngineKit joinChannelByToken:channelId:info:uid:joinSuccess:]), the SDK starts to establish a connection to the specified channel, triggers the [connectionChangedToState]([BrtcEngineDelegate rtcEngine:connectionChangedToState:reason:]) callback, and switches to the `BrtcConnectionStateConnecting` state.<br>
<br>
When the SDK successfully joins the channel, the SDK triggers the [connectionChangedToState]([BrtcEngineDelegate rtcEngine:connectionChangedToState:reason:]) callback and switches to the `BrtcConnectionStateConnected` state.<br>
<br>
After the SDK joins the channel and when it finishes initializing the media engine, the SDK triggers the [didJoinChannel]([BrtcEngineDelegate rtcEngine:didJoinChannel:withUid:elapsed:]) callback.
*/
    BrtcConnectionStateConnecting = 2,
    /** <p>3: The SDK is connected to Brtc's edge server and joins a channel. You can now publish or subscribe to a media stream in the channel.</p>
If the connection to the channel is lost because, for example, the network is down or switched, the SDK automatically tries to reconnect and triggers:
<li> The [rtcEngineConnectionDidInterrupted]([BrtcEngineDelegate rtcEngineConnectionDidInterrupted:])(deprecated) callback
<li> The [connectionChangedToState]([BrtcEngineDelegate rtcEngine:connectionChangedToState:reason:]) callback, and switches to the `BrtcConnectionStateReconnecting` state.
    */
    BrtcConnectionStateConnected = 3,
    /** <p>4: The SDK keeps rejoining the channel after being disconnected from a joined channel because of network issues.</p>
<li>If the SDK cannot rejoin the channel within 10 seconds after being disconnected from Brtc's edge server, the SDK triggers the [rtcEngineConnectionDidLost]([BrtcEngineDelegate rtcEngineConnectionDidLost:]) callback, stays in the `BrtcConnectionStateReconnecting` state, and keeps rejoining the channel.
<li>If the SDK fails to rejoin the channel 20 minutes after being disconnected from Brtc's edge server, the SDK triggers the [connectionChangedToState]([BrtcEngineDelegate rtcEngine:connectionChangedToState:reason:]) callback, switches to the `BrtcConnectionStateFailed` state, and stops rejoining the channel.
    */
    BrtcConnectionStateReconnecting = 4,
    /** <p>5: The SDK fails to connect to Brtc's edge server or join the channel.</p>
You must call [leaveChannel]([BrtcEngineKit leaveChannel:]) to leave this state, and call [joinChannelByToken]([BrtcEngineKit joinChannelByToken:channelId:info:uid:joinSuccess:]) again to rejoin the channel.<br>
<br>
If the SDK is banned from joining the channel by Brtc's edge server (through the RESTful API), the SDK triggers the [rtcEngineConnectionDidBanned]([BrtcEngineDelegate rtcEngineConnectionDidBanned:])(deprecated) and [connectionChangedToState]([BrtcEngineDelegate rtcEngine:connectionChangedToState:reason:]) callbacks.
    */
    BrtcConnectionStateFailed = 5,
};

/** Reasons for the connection state change. */
typedef NS_ENUM(NSUInteger, BrtcConnectionChangedReason) {
    /** 0: The SDK is connecting to Brtc's edge server. */
    BrtcConnectionChangedConnecting = 0,
    /** 1: The SDK has joined the channel successfully. */
    BrtcConnectionChangedJoinSuccess = 1,
    /** 2: The connection between the SDK and Brtc's edge server is interrupted.  */
    BrtcConnectionChangedInterrupted = 2,
    /** 3: The connection between the SDK and Brtc's edge server is banned by Brtc's edge server. */
    BrtcConnectionChangedBannedByServer = 3,
    /** 4: The SDK fails to join the channel for more than 20 minutes and stops reconnecting to the channel. */
    BrtcConnectionChangedJoinFailed = 4,
    /** 5: The SDK has left the channel. */
    BrtcConnectionChangedLeaveChannel = 5,
    /** 6: The specified App ID is invalid. Try to rejoin the channel with a valid App ID. */
    BrtcConnectionChangedInvalidAppId = 6,
    /** 7: The specified channel name is invalid. Try to rejoin the channel with a valid channel name. */
    BrtcConnectionChangedInvalidChannelName = 7,
    /** 8: The generated token is invalid probably due to the following reasons:
<li>The App Certificate for the project is enabled in Dashboard, but you do not use Token when joining the channel. If you enable the App Certificate, you must use a token to join the channel.
<li>The uid that you specify in the [joinChannelByToken]([BrtcEngineKit joinChannelByToken:channelId:info:uid:joinSuccess:]) method is different from the uid that you pass for generating the token. */
    BrtcConnectionChangedInvalidToken = 8,
    /** 9: The token has expired. Generate a new token from your server. */
    BrtcConnectionChangedTokenExpired = 9,
    /** 10: The user is banned by the server. */
    BrtcConnectionChangedRejectedByServer = 10,
    /** 11: The SDK tries to reconnect after setting a proxy server. */
    BrtcConnectionChangedSettingProxyServer = 11,
    /** 12: The token renews. */
    BrtcConnectionChangedRenewToken = 12,
    /** 13: The client IP address has changed, probably due to a change of the network type, IP address, or network port. */
    BrtcConnectionChangedClientIpAddressChanged = 13,
    /** 14: Timeout for the keep-alive of the connection between the SDK and Brtc's edge server. The connection state changes to BrtcConnectionStateReconnecting(4). */
    BrtcConnectionChangedKeepAliveTimeout = 14,
};

/** The state code in BrtcChannelMediaRelayState.
 */
typedef NS_ENUM(NSInteger, BrtcChannelMediaRelayState) {
    /** 0: The SDK is initializing.
     */
    BrtcChannelMediaRelayStateIdle = 0,
    /** 1: The SDK tries to relay the media stream to the destination channel.
     */
    BrtcChannelMediaRelayStateConnecting = 1,
    /** 2: The SDK successfully relays the media stream to the destination channel.
     */
    BrtcChannelMediaRelayStateRunning = 2,
    /** 3: A failure occurs. See the details in `error`.
     */
    BrtcChannelMediaRelayStateFailure = 3,
};

/** The event code in BrtcChannelMediaRelayEvent.
 */
typedef NS_ENUM(NSInteger, BrtcChannelMediaRelayEvent) {
    /** 0: The user disconnects from the server due to poor network connections.
     */
    BrtcChannelMediaRelayEventDisconnect = 0,
    /** 1: The network reconnects.
     */
    BrtcChannelMediaRelayEventConnected = 1,
    /** 2: The user joins the source channel.
     */
    BrtcChannelMediaRelayEventJoinedSourceChannel = 2,
    /** 3: The user joins the destination channel.
     */
    BrtcChannelMediaRelayEventJoinedDestinationChannel = 3,
    /** 4: The SDK starts relaying the media stream to the destination channel.
     */
    BrtcChannelMediaRelayEventSentToDestinationChannel = 4,
    /** 5: The server receives the video stream from the source channel.
     */
    BrtcChannelMediaRelayEventReceivedVideoPacketFromSource = 5,
    /** 6: The server receives the audio stream from the source channel.
     */
    BrtcChannelMediaRelayEventReceivedAudioPacketFromSource = 6,
    /** 7: The destination channel is updated.
     */
    BrtcChannelMediaRelayEventUpdateDestinationChannel = 7,
    /** 8: The destination channel update fails due to internal reasons.
     */
    BrtcChannelMediaRelayEventUpdateDestinationChannelRefused = 8,
    /** 9: The destination channel does not change, which means that the destination channel fails to be updated.
     */
    BrtcChannelMediaRelayEventUpdateDestinationChannelNotChange = 9,
    /** 10: The destination channel name is NULL.
     */
    BrtcChannelMediaRelayEventUpdateDestinationChannelIsNil = 10,
    /** 11: The video profile is sent to the server.
     */
    BrtcChannelMediaRelayEventVideoProfileUpdate = 11,
};

/** The error code in BrtcChannelMediaRelayError.
 */
typedef NS_ENUM(NSInteger, BrtcChannelMediaRelayError) {
    /** 0: The state is normal.
     */
    BrtcChannelMediaRelayErrorNone = 0,
    /** 1: An error occurs in the server response.
     */
    BrtcChannelMediaRelayErrorServerErrorResponse = 1,
    /** 2: No server response. You can call the [leaveChannel]([BrtcEngineKit leaveChannel:]) method to leave the channel.
     */
    BrtcChannelMediaRelayErrorServerNoResponse = 2,
    /** 3: The SDK fails to access the service, probably due to limited resources of the server.
     */
    BrtcChannelMediaRelayErrorNoResourceAvailable = 3,
    /** 4: The server fails to join the source channel.
     */
    BrtcChannelMediaRelayErrorFailedJoinSourceChannel = 4,
    /** 5: The server fails to join the destination channel.
     */
    BrtcChannelMediaRelayErrorFailedJoinDestinationChannel = 5,
    /** 6: The server fails to receive the data from the source channel.
     */
    BrtcChannelMediaRelayErrorFailedPacketReceivedFromSource = 6,
    /** 7: The source channel fails to transmit data.
     */
    BrtcChannelMediaRelayErrorFailedPacketSentToDestination = 7,
    /** 8: The SDK disconnects from the server due to poor network connections. You can call the [leaveChannel]([BrtcEngineKit leaveChannel:]) method to leave the channel.
     */
    BrtcChannelMediaRelayErrorServerConnectionLost = 8,
    /** 9: An internal error occurs in the server.
     */
    BrtcChannelMediaRelayErrorInternalError = 9,
    /** 10: The token of the source channel has expired.    
     */
    BrtcChannelMediaRelayErrorSourceTokenExpired = 10,
    /** 11: The token of the destination channel has expired.
     */
    BrtcChannelMediaRelayErrorDestinationTokenExpired = 11,
};

/** Network type. */
typedef NS_ENUM(NSInteger, BrtcNetworkType) {
  /** -1: The network type is unknown. */
  BrtcNetworkTypeUnknown = -1,
  /** 0: The SDK disconnects from the network. */
  BrtcNetworkTypeDisconnected = 0,
  /** 1: The network type is LAN. */
  BrtcNetworkTypeLAN = 1,
  /** 2: The network type is Wi-Fi (including hotspots). */
  BrtcNetworkTypeWIFI = 2,
  /** 3: The network type is mobile 2G. */
  BrtcNetworkTypeMobile2G = 3,
  /** 4: The network type is mobile 3G. */
  BrtcNetworkTypeMobile3G = 4,
  /** 5: The network type is mobile 4G. */
  BrtcNetworkTypeMobile4G = 5,
};

/** The video encoding degradation preference under limited bandwidth. */
typedef NS_ENUM(NSInteger, BrtcDegradationPreference) {
    /** (Default) Degrades the frame rate to guarantee the video quality. */
    BrtcDegradationMaintainQuality = 0,
    /** Degrades the video quality to guarantee the frame rate. */
    BrtcDegradationMaintainFramerate = 1,
    /** Reserved for future use. */
    BrtcDegradationBalanced = 2,
};
/** The lightening contrast level. */
typedef NS_ENUM(NSUInteger, BrtcLighteningContrastLevel) {
    /** Low contrast level. */
    BrtcLighteningContrastLow = 0,
    /** (Default) Normal contrast level. */
    BrtcLighteningContrastNormal = 1,
    /** High contrast level. */
    BrtcLighteningContrastHigh = 2,
};

/** The state of the probe test result. */
typedef NS_ENUM(NSUInteger, BrtcLastmileProbeResultState) {
  /** 1: the last-mile network probe test is complete. */
  BrtcLastmileProbeResultComplete = 1,
  /** 2: the last-mile network probe test is incomplete and the bandwidth estimation is not available, probably due to limited test resources. */
  BrtcLastmileProbeResultIncompleteNoBwe = 2,
  /** 3: the last-mile network probe test is not carried out, probably due to poor network conditions. */
  BrtcLastmileProbeResultUnavailable = 3,
};

/** The state of the local video stream. */
typedef NS_ENUM(NSInteger, BrtcLocalVideoStreamState) {
  /** 0: the local video is in the initial state. */
  BrtcLocalVideoStreamStateStopped = 0,
  /** 1: the local video capturer starts successfully. */
  BrtcLocalVideoStreamStateCapturing = 1,
  /** 2: the first local video frame encodes successfully. */
  BrtcLocalVideoStreamStateEncoding = 2,
  /** 3: the local video fails to start. */
  BrtcLocalVideoStreamStateFailed = 3,
};

/** The detailed error information of the local video. */
typedef NS_ENUM(NSInteger, BrtcLocalVideoStreamError) {
  /** 0: the local video is normal. */
  BrtcLocalVideoStreamErrorOK = 0,
  /** 1: no specified reason for the local video failure. */
  BrtcLocalVideoStreamErrorFailure = 1,
  /** 2: no permission to use the local video device. */
  BrtcLocalVideoStreamErrorDeviceNoPermission = 2,
  /** 3: the local video capturer is in use. */
  BrtcLocalVideoStreamErrorDeviceBusy = 3,
  /** 4: the local video capture fails. Check whether the capturer is working properly. */
  BrtcLocalVideoStreamErrorCaptureFailure = 4,
  /** 5: the local video encoding fails. */
  BrtcLocalVideoStreamErrorEncodeFailure = 5,
};
