//
//  BRTCMediaPlayerManager.h
//  BRTC
//
//  Created by lw0717 on 2024/8/30.
//  Copyright © 2024 BRTC. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <BRTC/BRTCDef.h>

NS_ASSUME_NONNULL_BEGIN

/**
* @brief 播放器的状态
*/
typedef NS_ENUM(NSUInteger, BRTCMediaPlayerState) {
    BRTCMediaPlayerState_IDLE,            ///< 空闲, 播放器会在你打开媒体文件之前和结束播放之后返回该状态码
    BRTCMediaPlayerState_OPENING,         ///< 正在打开媒体文件
    BRTCMediaPlayerState_OPEN_COMPLETED,  ///< 成功打开媒体文件
    BRTCMediaPlayerState_PLAYING,         ///< 正在播放
    BRTCMediaPlayerState_PAUSED,          ///< 暂停播放
    BRTCMediaPlayerState_COMPLETED,       ///< 播放完毕
    BRTCMediaPlayerState_STOPPED,         ///< 播放已停止
    BRTCMediaPlayerState_FAILED           ///< 播放失败
};

/**
 * @brief 播放器状态改变的原因
 */
typedef NS_ENUM(NSUInteger, BRTCMediaPlayerReason) {
  BRTCMediaPlayerReason_NONE = 0,              ///< 没有错误
  BRTCMediaPlayerReason_INVALID_ARGUMENTS,     ///< 无效的参数
  BRTCMediaPlayerReason_OBJ_NOT_INITIALIZED,   ///< 对象未初始化
  BRTCMediaPlayerReason_SRC_BUFFER_UNDERFLOW,  ///< 播放缓冲区数据不足
  BRTCMediaPlayerReason_CODEC_NOT_SUPPORTED,   ///< 解码器不支持该 codec
  BRTCMediaPlayerReason_UNKNOWN,               ///< 未知错误
};

/**
 * @brief 播放器的事件类型
 */
typedef NS_ENUM(NSUInteger, BRTCMediaPlayerEvent) {
  BRTCMediaPlayerEvent_FIRST_DISPLAYED,  ///< 视频首帧出图
  BRTCMediaPlayerEvent_BUFFER_LOW,       ///< 当前缓冲的数据不足以支持播放
  BRTCMediaPlayerEvent_BUFFER_RECOVER,   ///< 当前缓冲的数据刚好能支持播放
  BRTCMediaPlayerEvent_SEEK_COMPLETE,    ///< seek 完成
  BRTCMediaPlayerEvent_SEEK_ERROR,       ///< seek 出错
};

/**
 * @brief 播放器媒体流信息
 */
@interface BRTCPlayerStreamInfo : NSObject

/// 是否有视频
@property (nonatomic, assign) BOOL hasVideo;
/// 是否有音频
@property (nonatomic, assign) BOOL hasAudio;
/// 视频宽
@property (nonatomic, assign) int videoWidth;
/// 视频高
@property (nonatomic, assign) int videoHeight;
/// 音频采样率
@property (nonatomic, assign) int audioSampleRate;
/// 音频声道
@property (nonatomic, assign) int audioChannels;

@end

/**
 * @brief 播放器音视频统计信息
 *
 */
@interface BRTCMediaPlayerStats : NSObject

/// 视频帧率
@property (nonatomic, assign) int frameRate;
/// 视频码率，单位Kbps
@property (nonatomic, assign) int videoBitrate;
/// 音频码率，单位Kbps
@property (nonatomic, assign) int audioBitrate;
/// 总码率，单位Kbps
@property (nonatomic, assign) int totalBitrate;

@end

/**
 * @brief BRTCMediaPlayerSource
 *
 */
@interface BRTCMediaPlayerSource : NSObject

/// 播放器 ID
/// 【特殊说明】如果需要播放多路媒体资源，需要使用 ID, 平台语言层分配唯一 ID
/// 目前暂不支持多路媒体资源同时播放，等多流版本发版后才能支持
@property (nonatomic, assign) int playerId;

/// 【字段含义】媒体路径，本地文件的绝对路径 或者 网络媒体 url
@property (nonatomic, copy) NSString *path;

/// 【字段含义】媒体循环播放的次数
/// 【取值含义】 0: 表示播放一次；1 表示播放两次；以此类推
///            -1: 无限循环播放。
/// 【默认取值】 0: 表示播放一次
@property (nonatomic, assign) int loopCount;

/// 【字段含义】是否将媒体传到远端
/// 【取值含义】true：
/// 媒体在本地播放的同时，会上行至云端，因此远端用户也能看到听到该媒体；
///            false：媒体不会上行至云端，因此只能在本地看到听到该媒体。
/// 【默认取值】false
@property (nonatomic, assign) BOOL publish;

/// 【字段含义】媒体开始播放时间点，单位毫秒
@property (nonatomic, assign) long startTimeMS;

/// 【字段含义】打开媒体文件后，是否开启自动播放
/// 【取值含义】true：开启自动播放；false：关闭自动播放。
/// 【默认取值】true, 不用再调用 play 接口
@property (nonatomic, assign) BOOL autoPlay;

/// 【字段含义】播放器中的流推送到 rtc 房间内编码的宽度
@property (nonatomic, assign) int videoEncodeWidth;

/// 【字段含义】播放器中的流推送到 rtc 房间内编码的高度
@property (nonatomic, assign) int videoEncodeHeight;

/// 【字段含义】播放器中的流推送到 rtc 房间内编码的帧率
@property (nonatomic, assign) int videoEncodeFps;

/// 【字段含义】播放器中的流推送到 rtc 房间内编码的码率
@property (nonatomic, assign) int videoEncodeBitrate;

@end


/**
 * @brief 媒体播放器事件回调
 *
 */
@protocol BRTCMediaPlayerDelegate <NSObject>
@optional

/**
 * 播放器状态变更回调
 *
 * @param state   状态码
 * @param reason  原因码
 * @param msg     附加信息
 */
- (void)onPlayerStateChanged:(int)playerId
                       state:(BRTCMediaPlayerState)state
                      reason:(BRTCMediaPlayerReason)reason
                         msg:(NSString *)msg;

/**
 * 当前媒体资源播放进度回调
 *
 * @note SDK 每隔 1 秒会自动触发该回调
 *
 * @param curPtsMS   当前播放进度，单位毫秒
 * @param durationMS  当前媒体资源总时长，单位毫秒
 */
- (void)onPlayProgress:(int)playerId curPtsMS:(long)curPtsMS durationMS:(long)durationMS;

/**
 * 当前缓冲数据能播放的时间
 *
 * @note SDK 每隔 1 秒会自动触发该回调
 *       当缓冲数据支持的播放时间小于 0 时，返回 PLAYER_EVENT_BUFFER_LOW 事件
 *       当缓冲数据支持的播放时间大于 0 时，返回 PLAYER_EVENT_BUFFER_RECOVER
 * 事件
 *
 * @param playCachedBuffer 当前缓冲的数据能支持的播放时间，单位毫秒
 */
- (void)onPlayBufferUpdated:(int)playerId playCachedBuffer:(long)playCachedBuffer;

/**
 * 播放器事件回调
 *
 * @param event 事件类型
 * @param elapsedTime 事件发生的时间，单位毫秒
 * @param msg 附加信息
 */
- (void)onPlayerEvent:(int)playerId
                event:(BRTCMediaPlayerEvent)event
          elapsedTime:(long)elapsedTime
                  msg:(NSString *)msg;

/**
 * 播放器媒体统计信息
 *
 * @note SDK 每隔 2 秒会自动触发该回调
 *
 * @param stats 媒体流信息，主要包括码率、帧率等信息
 */
- (void)onPlayerMediaStats:(int)playerId stats:(BRTCMediaPlayerStats *)stats;

/**
 * 播放器媒体流信息
 *
 * @param info 媒体流信息，主要包括宽高、音频采样率、声道数等
 */
- (void)onStreamInfo:(int)playerId info:(BRTCPlayerStreamInfo *)info;

@end

@interface BRTCMediaPlayerManager : NSObject

- (instancetype)init NS_UNAVAILABLE;

/**
 * 播放状态回调 delegate
 */
@property (nonatomic, weak) id<BRTCMediaPlayerDelegate> delegate;

/**
 * 设置视频显示窗口
 *
 * @param playerId 播放器 ID
 * @param view 承载视频画面的控件
 *
 * @return  0: 设置成功
 *         <0: 设置失败
 */
- (int)setVideoRenderer:(int)playerId
                   view:(BRTCView *)view;

/**
 * 打开媒体资源
 *
 * @param source 媒体资源参数
 *
 * @return  0: 设置成功
 *         <0: 设置失败
 */
- (int)open:(BRTCMediaPlayerSource *)source;

/**
 * 开始播放
 *
 * @note 必须等待 onPlayerStateChanged 回调 BRTCMediaPlayerState_OPEN_COMPLETED 后才可以调用
 *
 * @param playerId  播放器 ID
 *
 * @return  0: 播放成功
 *         <0: 播放失败
 */
- (int)play:(int)playerId;

/**
 * 停止播放
 *
 * @note 停止播放后，如需重新播放，需要重新调用 open, 非 autoPlay
 * 的，还需要再调用 play 开始播放
 *
 * @param playerId  播放器 ID
 *
 * @return  0: 停止成功
 *         <0: 停止失败
 */
- (int)stop:(int)playerId;

/**
 * 定位到媒体文件的指定播放位置（单位：毫秒）
 *
 * @note 如果在播放过程中调用该接口，SDK 会自动从指定位置开始播放
 *       如果在暂停状态下调用该接口，SDK
 * 会定位到您指定的位置，如需播放，请调用 resume
 *
 * @param playerId  播放器 ID
 * @param newPos 指定的播放位置 单位毫秒
 *
 * @return  0: 设置成功
 *         <0: 设置失败
 */
- (int)seek:(int)playerId newPos:(long)newPos;

/**
 * 暂停播放
 *
 * @param playerId  播放器 ID
 *
 * @return  0: 暂停成功
 *         <0: 暂停失败
 */
- (int)pause:(int)playerId;

/**
 * 恢复播放
 *
 * @param playerId  播放器 ID
 *
 * @return  0: 恢复成功
 *         <0: 恢复失败
 */
- (int)resume:(int)playerId;

/**
 * 设置是否静音
 *
 * @param playerId  播放器 ID
 * @param muted true: 静音  false: 取消静音
 *
 * @return  0: 设置成功
 *         <0: 设置失败
 */
- (int)muteAudio:(int)playerId muted:(BOOL)muted;

/**
 * 同时设置该媒体中音频的本地音量和远端音量的大小
 *
 * - 本地音量：即主播本地可以听到的播放音量大小。
 * - 远端音量：即观众端可以听到的音频的音量大小。
 *
 * @param playerId  播放器 ID
 * @param volume 音量大小，取值范围为0 - 100，默认值：100。
 *
 * @note 如果将 volume 设置成 100 之后感觉音量还是太小，可以将 volume
 * 最大设置成 150，但超过 100 的 volume 会有爆音的风险，请谨慎操作。
 *
 * @return  0: 设置成功
 *         <0: 设置失败
 */
- (int)setAllAudioVolume:(int)playerId volume:(int)volume;

/**
 * 设置播放器中音频在远端听到的音量的大小
 *
 * - 观众端可听到的音频的音量大小。
 *
 * @param playerId  播放器 ID
 * @param volume 音量大小，取值范围为0 - 100；默认值：100
 *
 * @note 如果将 volume 设置成 100 之后感觉音量还是太小，可以将 volume
 * 最大设置成 150，但超过 100 的 volume 会有爆音的风险，请谨慎操作。
 * @return  0: 设置成功
 *         <0: 设置失败
 */
- (int)setAudioPublishVolume:(int)playerId volume:(int)volume;

/**
 * 设置播放器中音频在本地播放音量的大小
 *
 * @param playerId  播放器 ID
 * @param volume 音量大小，取值范围为0 - 100，默认值：100。
 *
 * @note 如果将 volume 设置成 100 之后感觉音量还是太小，可以将 volume
 * 最大设置成 150，但超过 100 的 volume 会有爆音的风险，请谨慎操作。
 *
 * @return 0: 设置成功
 *        <0: 设置失败
 */
- (int)setAudioPlayoutVolume:(int)playerId volume:(int)volume;

/**
 * 获取当前播放进度（单位：毫秒）
 *
 * @param playerId  播放器 ID
 *
 * @return 成功返回当前播放时间，单位：毫秒，失败返回-1
 */
- (long)getCurrentPlayPosInMS:(int)playerId;

/**
 * 获取媒体文件总时长（单位：毫秒）
 *
 * @param playerId  播放器 ID
 *
 * @return 成功返回时长，失败返回-1
 */
- (long)getDurationInMS:(int)playerId;

@end

NS_ASSUME_NONNULL_END
